/*
 * Copyright (c) 2011, Takashi TOYOSHIMA <toyoshim@gmail.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of the authors nor the names of its contributors may be
 *   used to endorse or promote products derived from this software with out
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUE
 * NTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include <avr/io.h>
#include "uart.h"
#include "sram.h"
#include "sdcard.h"
//#include "eeprom.h"
#include "config.h"
#include "platform.h"
#include "i8080.h"


void boot (unsigned short volume) {
  i8080_reset();
  sram_bank(0);
  int j;
  unsigned long image = (unsigned long) volume;
  sdcard_fetch(image << 14);	// read bootsector
  for (j = 0; j < 512; j++) sram_write(j, sdcard_read(j));
  sram_write(0x100, volume);
  do {
  } while (0 == i8080_run());	// start virtual machine at bootloader
}

#if defined(MONITOR)
int strdcmp (char *s1, char *s2, char s) {
  for (;;) {
    if (((0 == *s1) || (s == *s1)) && 
        ((0 == *s2) || (s == *s2))) return 0;
    if (*s1++ != *s2++) return -1;
  }
}

#if defined(MON_MEM) | defined(MON_SDC)
int strndcmp (char *s1, char *s2, int n, char s) {
  int i;
  for (i = 0; i < n; i++) {
    if (((0 == *s1) || (s == *s1)) ||
        ((0 == *s2) || (s == *s2))) return -1;
    if (*s1++ != *s2++) return -1;
  }
  return 0;
}

int getnum (char *s) {
  int rc = 0;
  if (0 == strndcmp(s, "0x", 2, 0)) {
    s += 2;
    for (;;) {
      if (0 == *s) return rc;
      rc *= 16;
      if (('0' <= *s) && (*s <= '9')) rc += (*s - '0');
      else if (('a' <= *s) && (*s <= 'f')) rc += (*s - 'a' + 10);
      else if (('A' <= *s) && (*s <= 'F')) rc += (*s - 'A' + 10);
      else return rc / 16;
      s++;
    }
  } else {
    for (;;) {
      if (0 == *s) return rc;
      rc *= 10;
      if (('0' <= *s) && (*s <= '9')) rc += (*s - '0');
      else return rc / 10;
      s++;
    }
  }
  return rc;
}
#endif // defined(MON_MEM) | defined(MON_SDC)


char *split (char *s, char c) {
  for (; 0 != *s; s++) if (c == *s) return ++s;
  return NULL;
}

void prompt (void) {
  char buffer[MAX_PROMPT + 1];
  unsigned char size = 0;
  uart_puts("CP/Mega88>");
  for (;;) {
    int c;
    do {
      c = uart_getchar();
    } while (c < 0);
    if ((0x08 == c) || (0x7f == c)) {
      if (0 != size) {
        size--;
        uart_putchar(0x08);
        uart_putchar(0x20);
        uart_putchar(0x08);
      }
    } else if (0x0a == c) {
    } else if (0x0d == c) {
      uart_putsln("");
      break;
    } else if ((0 == size) && (0x10 == c)) {
      char *p;
      for (p = buffer; 0 != *p; p++) size++;
      uart_puts(buffer);
    } else {
      if (MAX_PROMPT != size) {
        buffer[size++] = c;
        uart_putchar(c);
      }
    }
  }
  buffer[size] = 0;
  char *cmd = buffer;
  char *arg = split(cmd, ' ');
  if (0 == *cmd) {
    return;
  } else if (0 == strdcmp("r", cmd, 0)) {
    platform_reset();
  } else if (0 == strdcmp("h", cmd, 0)) {
    goto usage;
  } else if (0 == strdcmp("b", cmd, ' ')) {
    unsigned short n;
    n = getnum(arg);
    if ((n < 2) || (n > 252)) goto huh;
    boot(n);
    return;
//  } else if (0 == strdcmp("a", cmd, ' ')) {
//    if (NULL == arg)
//			eeprom_write(8,0);
//		else {
//    	unsigned short n;
//    	n = getnum(arg);
//			if ((n < 2) || (n > 254))
//    		eeprom_write(8, n);
//			else
//				eeprom_write(8,0);
//			}
//    return;
#if defined(MON_MEM)
  } else if (0 == strdcmp("md", cmd, ' ')) {
    if (NULL == arg) goto huh;
    unsigned short n;
    n = getnum(arg) & 0xFFF0;
    int i;
    for (i = 0; i < 256; i++) {
      char buf[17];
      if (0 == (i & 15)) {
        uart_puthex((n+i)>>8);
        uart_puthex(n+i);
        uart_puts(" ");
      }      
      unsigned char c = sram_read(n+i);
      uart_puts(" ");
      uart_puthex(c);
      if ((c < '!') || ('~' < c)) c = '.';
      buf[i & 15] = c;
      if (15 == (i & 15)) {
        buf[16] = 0;
        uart_puts(": ");
        uart_puts(buf);
        uart_putsln(" ");
      }
    }
    return;
  } else if (0 == strdcmp("mw", cmd, ' ')) {
    if (NULL == arg) goto huh;
    char *addr = arg;
    char *data = split(addr, ',');
    if (NULL == data) goto huh;
    unsigned short n_addr;
    unsigned short n_data;
    n_addr = getnum(addr);
    n_data = getnum(data);
    sram_write(n_addr, n_data);
    return;
  } else if (0 == strdcmp("mp", cmd, ' ')) {
    if (NULL == arg) goto huh;
    unsigned short n;
    n = getnum(arg);
    if (n > 3) goto huh;
    sram_bank(n);
    return;
#endif // defined(MON_MEM)
#if defined (MON_SDC)
  } else if (0 == strdcmp("so", cmd, 0)) {
    uart_puts("SDC: ");
    char rc = sdcard_open();
    if (-1 == rc) uart_putsln("ERR");
    else if (rc == -64) uart_putsln("HC");
    else  uart_putsln("SC");
    return;
  } else if (0 == strdcmp("sd", cmd, 0)) {
    uart_putsln("");
    int i;
    for (i = 0; i < 512; i++) {
      char buf[17];
      if (0 == (i & 15)) {
        uart_puthex(i>>8);
        uart_puthex(i);
        uart_puts(" ");
      } 
      unsigned char rc = sdcard_read(i);
      uart_puts(" ");
      uart_puthex(rc);
      if ((rc < '!') || ('~' < rc)) rc = '.';
      buf[i & 15] = rc;
      if (15 == (i & 15)) {
        buf[16] = 0;
        uart_puts("  ");
        uart_puts(buf);
        uart_putsln(" ");
      }
    }
    uart_puts("crc: ");
    unsigned short crc = sdcard_crc();
    uart_puthex(crc >> 8);
    uart_puthex(crc);
    uart_putsln("");
    return;


  } else if (0 == strdcmp("sw", cmd, ' ')) {
    if (NULL == arg) goto huh;
    char *offs = arg;
    char *data = split(offs, ',');
    if (NULL == data) goto huh;
    unsigned short n_offs;
    unsigned short n_data;
    n_offs = getnum(offs);
    n_data = getnum(data);
    sdcard_write(n_offs, n_data);
    return;
#endif // defined(MON_MEM)
#if defined (MON_SDC)
  } else if (0 == strdcmp("sf", cmd, ' ')) {
    if (NULL == arg) goto huh;
    char *volume = arg;
    char *sector = split(volume, ',');
    if (NULL == sector) goto huh;
    unsigned long n_volume;
    unsigned short n_sector;
    n_volume = getnum(volume);
    n_sector = getnum(sector);
    char rc = sdcard_fetch((n_volume << 14) + n_sector);
    if (rc >= 0) uart_putsln(" ok");
    else {
      uart_puts(" error(");
      uart_puthex(-rc);
      uart_putsln(")");
    }
    return;
  } else if (0 == strdcmp("ss", cmd, ' ')) {
    if (NULL == arg) goto huh;
    char *volume = arg;
    char *sector = split(volume, ',');
    if (NULL == sector) goto huh;
    unsigned long n_volume;
    unsigned short n_sector;
    n_volume = getnum(volume);
    n_sector = getnum(sector);
    char rc = sdcard_store((n_volume << 14) + n_sector);
    if (rc >= 0) uart_putsln(" ok");
    else {
      uart_puts(" error(");
      uart_puthex(-rc);
      uart_putsln(")");
    }
    return;
#endif // defined(MON_SDC)
  }
  
huh:
  uart_putsln("Huh??"); // Tribute to ADV
  return;
  
usage:
#if defined(MON_HELP)
# if !defined(MSG_MIN)
  uart_putsln("monitor commands");
  uart_putsln(" r			: reset");
  uart_putsln(" b <vol>			: boot image");
//  uart_putsln(" a <on/off>		: auto boot");
#  if defined(MON_MEM)
  uart_putsln(" md <addr>		: memory dump from <addr>");
  uart_putsln(" mw <addr>,<data>	: memory write <data> to <addr>");
  uart_putsln(" mp <0-3>		: select 32k memory page");
#  endif // defined(MON_MEM)
#  if defined(MON_SDC)
  uart_putsln(" so			: sdcard open");
  uart_putsln(" sf <vol>,<sect>		: sdcard fetch block");
  uart_putsln(" sd			: sdcard dump block buffer");
  uart_putsln(" sw <offs>,<data>	: sdcard write block buffer");
  uart_putsln(" ss <vol>,<sect>		: sdcard store block");
#  endif // defined(MON_SDC)
# else // !defined(MSG_MIN)
  uart_putsln("    Commands:");
  uart_putsln("	R	reset");
  uart_putsln("	B v	boot");
//  uart_putsln("	A n	auto");
#  if defined(MON_MEM)
  uart_putsln("    Memory:");
  uart_putsln("	MD a	dump");
  uart_putsln("	MW a,d	write");
  uart_putsln("	MP n	page");
#  endif // defined(MON_MEM)
#  if defined(MON_SDC)
  uart_putsln("    SD Card:");
  uart_putsln("	SO	open");
  uart_putsln("	SF v,s	fetch");
  uart_putsln("	SD	dump");
  uart_putsln("	SW o,d	write");
  uart_putsln("	SS v,s	store");
#  endif // defined(MON_SDC)
  uart_putsln("");
# endif // !defined(MSG_MIN)
#endif // defined(MON_HELP)
  return;
}
#endif // defined(MONITOR)

#if defined(CLR_MEM)
void mem_clr (void) {
  unsigned short addr;
  addr = 0;
  do {
    sram_write(addr++, 0);
  } while (0 != addr);
}
#endif // defined(CLR_MEM)

#if defined(MON_MEM) | defined(CHK_MEM)
void mem_chk (void) {
  unsigned char test;
  unsigned short addr = 0;
  for (test = 0; test < 2; addr++) {
    unsigned char c;
    unsigned char err = 0;
    if (0 == test) {
      sram_write(addr, 0xaa);
      c = sram_read(addr);
      if (0xaa != c) err |= 1;
#if !defined(CHK_MIN)
      sram_write(addr, 0x55);
      c = sram_read(addr);
      if (0x55 != c) err |= 2;
#endif // !defined(CHK_MIN)
      if (0 == (addr & 1)) sram_write(addr, addr);
      else sram_write(addr, addr >> 8);
#if !defined(CHK_MIN)
      c = sram_read(addr);
      if ((0 == (addr & 1)) & ((addr & 0xff) != c)) err |= 4;
      if ((0 != (addr & 1)) & ((addr >> 8) != c)) err |= 4;
#endif // !defined(CHK_MIN)
    } else {
      c = sram_read(addr);
      if ((0 == (addr & 1)) & ((addr & 0xff) != c)) err |= 4;
      if ((0 != (addr & 1)) & ((addr >> 8) != c)) err |= 4;
    }
    if (0 != err) {
#if defined(CHK_MIN)
      uart_puts("ERR: ");
#else // defined(CHK_MIN)
      if (0 == test) uart_puts("write failed at 0x");
      else uart_puts("address failed at 0x");
#endif // defined(CHK_MIN)
      uart_puthex(addr >> 8);
      uart_puthex(addr);
#if defined(CHK_MIN)
      uart_putsln("");
#else // defined(CHK_MIN)
      uart_puts(" (");
      if (0 == test) uart_puthex(err);
      else uart_puthex(c);
      uart_putsln(")");
#endif // defined(CHK_MIN)
      return;
    }
    if (0xfff == (addr & 0xfff)) {
#if defined(MSG_MIN)
# if defined(CHK_MIN)
      if (0 != test) uart_puts("MEM: ");
# else // defined(CHK_MIN)
      if (0 == test) uart_puts("mem wrt: ");
      else uart_puts("mem adr: ");
# endif // defined(CHK_MIN)
#else // defined(MSG_MIN)
      if (0 == test) uart_puts("memory write test: ");
      else uart_puts("memory address test: ");
#endif // defined(MSG_MIN)
#if defined(CHK_MIN)
      if (0 != test) {
        uart_putnum_u16(addr, 5);
        uart_puts("/65535\r");
      }
#else // defined(CHK_MIN)
      uart_putnum_u16(addr, 5);
      uart_puts("/65535\r");
#endif // defined(CHK_MIN)
    }
    if (0xffff == addr) {
      test++;
#if defined(CHK_MIN)
      if (0 != test) uart_puts("\n");
#else // !defined(CHK_MIN)
      uart_puts("\n");
#endif // !defined(CHK_MIN)
    }
  }
}
#endif // defined(MON_MEM) | defined(CHK_MEM)

// defining CP/M Console, Reader, Punch, List and Aux devices
// as well as Disk and DMA I/O

void out (unsigned char port, unsigned char val) {
  static unsigned long volume = 0;
  static unsigned char trk_lo = 0;
  static unsigned short trk_hi = 0;
  static unsigned char sect = 0;
  static unsigned long cur_blk = 0xFFFFFFFF;
  static unsigned char active = 0;
  static unsigned char dma_lo = 0;
  static unsigned short dma_hi = 0;

  switch(port) {
  case 1:			// Console data port
     uart_putchar(val);
     break;
  case 10:			// Set disk number
    volume = (unsigned long) val;
    break;
  case 11:			// Set Track Low byte
    trk_lo = val;
    break;
  case 12:			// Set track High byte
    trk_hi = (unsigned short) val;
    break;
  case 13:			// Set Sector
    sect = val;
    break;
  case 14: {  			// Perform disk I/O with values set above
    unsigned int track = trk_hi + trk_lo;
    unsigned long blk = (volume << 14) + (track << 5) + (sect >> 2);
    if (blk != cur_blk) {
      if (1 == active) {
        sdcard_flush();
        active = 0;
      }
      sdcard_fetch(blk);
      cur_blk = blk;
    }
    unsigned short off = (sect & 0x03) << 7;
    unsigned short i;
    unsigned short addr = (dma_hi << 8) + dma_lo;
    if (0 == val) {
      for (i = 0; i < 128; i++) sram_write(addr + i, sdcard_read(off + i));
    } else {
      for (i = 0; i < 128; i++) sdcard_write(off + i, sram_read(addr + i));
      active = 1;
      if (track == 1) {		// dir write
        sdcard_flush();
        active = 0;
      }
    }
    break; }
  case 16:			// Set DMA Buffer address Low byte
    dma_lo = val;
    break;
  case 17:			// Set DMA Buffer address High byte
    dma_hi = val;
    break;
  case 18:
    sram_bank(val);		// Set 32k memory bank 0-3
    break;
  }
}

unsigned char in (unsigned char port) {
  if (0 == port) {		// Console Status port
    if (0 != uart_peek()) return 0xff;
    return 0;
  } else if (1 == port) {	// Console Data port
    int c;
    do {
      c = uart_getchar();
    } while (-1 == c);
    return c;
  } else if (15 == port) {	// Disk Status Port
    return 0;
  }
  return 0;
}

int machine_boot (void) {
  uart_init();
  sram_init();
  sdcard_init();
  uart_putsln("\r\nbooting CP/Mega88 done.");
  mem_clr();
  mem_chk();
  uart_puts("SDC: ");
  char rc = sdcard_open();
  if (-1 == rc) uart_putsln("ERR");
  else if (rc == -64) uart_putsln("SDHC");
  else  uart_putsln("SDSC");
//  uart_puts("EEPROM: ");
//  if (0x0 != eeprom_read(0)) {
//    uart_putsln("init");
//    int i;
//    for (i = 0; i < 256; i++) eeprom_write(i, 0);
//  } else {
//    uart_putsln("load");
//    boot(eeprom_read(8));
//  }
  for (;;) prompt();
  return 0;
}
