;***********************************
;*  Z80 Test Prototype             *
;*  SECOND BOOT Program            *
;*  Andrew Lynch                   *
;*  lynchaj@yahoo.com              *
;*  2 Feb 2007                     *
;***********************************
             .ORG    $F800          ; RAM VARIABLES

INT_COUNTER: .DS     1              ; Interrupt-counter
COUNTER      .DS     1              ; Cycle counter main loop
SER_ON:      .DS     1              ; serial on/off
UART_FAIL:   .DS     1              ; UART has failed detection flag
SER_BAUD:    .DS     1              ; specify desired UART com rate in bps

;********************* HARDWARE IO ADR ************************************

; OBJECTIVE OF FIRST BOOT PROGRAM is to verify CPU, ROM, RAM, memory decode, and
; IO decode logic function

; Simple output are LEDs attached to chipselect logic at IO port and memory
; chipselect lines.  Chipselect lines are active when grounded and result
; in LEDs slightly dimming

IO_Y0:        .EQU    $60
IO_Y1:        .EQU    $68
IO_Y2:        .EQU    $70
IO_Y3:        .EQU    $78

; Simple input is the RESET key and the INTERRUPT key.  The RESET key starts
; and restarts the program dimming the LEDs in a cycle.  Pressing the INTERRUPT
; key halts the CPU and stops the program.  Pressing RESET again restarts.

; OBJECTIVE OF SECOND BOOT PROGRAM is to expand on FIRST BOOT program and 
; verify correct operation of PIO and UART circuits


; PIO 82C55 I/O
PIO1A:       .EQU    $60            ; (INPUT)  IN 1-8
PIO1B:       .EQU    $61            ; (OUTPUT) OUT TO LEDS
PIO1C:       .EQU    $62            ; (INPUT)  
PIO1CONT:    .EQU    $63            ; CONTROL BYTE PIO 82C55

; UART 16C550 SERIAL
UART0:       .EQU    $68            ; DATA IN/OUT
UART1:       .EQU    $69            ; CHECK RX
UART2:       .EQU    $6A            ; INTERRUPTS
UART3:       .EQU    $6B            ; LINE CONTROL
UART4:       .EQU    $6C            ; MODEM CONTROL
UART5:       .EQU    $6D            ; LINE STATUS
UART6:       .EQU    $6E            ; MODEM STATUS
UART7:       .EQU    $6F            ; SCRATCH REG.


;********************* CONSTANTS ****************************************
RAMTOP:      .EQU    $FFFF          ; 2Kb RAM   F800H-FFFFH

END:         .EQU    $FF            ; Mark END OF TEXT



;*******************************************************************
;*        START AFTER RESET,                                       *
;*        Function....: ready system and restart                   *
;*******************************************************************
             .ORG    0
             DI                    ; Disable interrupt
             LD     SP,RAMTOP      ; Set stack pointer to top off ram
             IM     1              ; Set interrupt mode 1
             JP     $100           ; jump to Start of program


;************************************************************************
;*        INTERRUPT-PROGRAM                                             *
;*        Function....:                                                 *
;*        Input.......:                                                 *
;*        Output......: number interrupt at adr. INT_COUNTER            *
;*        uses........: only alternative registres.                     *
;*        calls.......: none                                            *
;*        info........: TST.  date: 2 Feb 2007                          *
;************************************************************************
             .ORG   $38            ; Int mode 1
             DI                    ; disable
             EXX                   ; IN THE INT ROUTINE, YOU ONLY USES
             EX     AF,AF'         ; THE EXTRA REGISTERS.

             LD     A,(INT_COUNTER)
             INC    A
             LD     (INT_COUNTER),A

             EX     AF,AF'         ; BEFORE RETURN, SWITCH REGISTERS BACK
             EXX
             EI                    ; enable again
             RETI                  ; return from interrupt



             .ORG   $66            ; HERE IS THE NMI ROUTINE
             LD     HL,TXT_HELLO  ; POINT AT TEXT
             CALL   TX_SER        ; SENT TEXT 
             RETI

;*******************************************************************
;*        MAIN PROGRAM                                             *
;*******************************************************************
             .ORG   $100

             CALL   INIT_PIO    ; program the PIO 
             LD     A,0   
             OUT    (PIO1B),A   ; ALL BITS OFF

             LD     BC,10000    
             CALL   PAUSE

             LD     A,255   
             OUT    (PIO1B),A    ; ALL BITS ON FOR 1 SEC..

             LD     BC,10000
             CALL   PAUSE

             LD     A,0   
             OUT    (PIO1B),A    ; ALL BITS OFF

             LD     A,0   
             LD     (COUNTER),A  ; Initialize Cycle Counter

             LD     A,0   
             LD     (SER_ON),A   ; Initialize "Serial On" flag
                                 ; assume it is off until UART
                                 ; is autodetected
             LD     A,1   
             LD     (UART_FAIL),A   ; Initialize "UART FAIL" flag
                                 ; assume it has failed until UART
                                 ; is autodetected as working


                                   ; INIT AND TEST OF UART
                                   ; ONLY CALL THIS IF YOU
                                   ; HAVE THE UART MOUNTED..

             LD     A,$04          ; specify baud rate 9600 bps
             LD     (SER_BAUD),A   ; 9600,8,None,1
             CALL   INIT_UART      ; WITH NO FLOW CONTROL on terminal!!

             EI                    ; Start INT COUNTER

MAIN_LOOP:
             LD     A,(COUNTER)    ; GET COUNTER value from memory location
             OUT    (PIO1B),A      ; OUTPUT IT TO SOME LEDS

             LD     BC,10000       ; Wait for one second
             CALL   PAUSE

             INC A                 ; increment the cycle counter
             LD (COUNTER),A        ; and store it back in memory location


             LD     HL,TXT_HELLO   ; POINT AT TEXT
             CALL   TX_SER         ; SENT TEXT 

             CALL NEW_LINE         ; CR,LF on terminal

INNER_LOOP:

             LD     A,(SER_ON)     ; IF COM IS OFF
             CP     0              ; 
             JP     Z,MAIN_LOOP    ; skip the UART send/receive test

             CALL RX_SER_CHAR      ; get a character from UART (received char in B)

             CALL TX_SER_CHAR      ; output same character to UART (transmit char in B)

             LD A,B                ; put received character in A from B
                                   ; for output to PIO LEDs and
                                   ; comparison operations 

             OUT    (PIO1B),A      ; OUTPUT received char TO SOME LEDS

             LD     BC,10000       ; Wait for one second
             CALL   PAUSE

             CP $0D                ; is character a Carriage Return
             JP Z, CR_LF           ; print a NEW_LINE on terminal
             CP $31                ; is character a '1'
             JP Z, END_PROGRAM     ; if YES, end program (HALT CPU)
             CP $32                ; is character a '2'
             JP Z, MAIN_LOOP       ; if YES, exit UART send/receive test

                                   ; and continue with main loop
             JP INNER_LOOP         ; if NO, try again with UART
                                   ; send/receive test
             


             JP MAIN_LOOP


END_PROGRAM:
             HALT


CR_LF:
             CALL NEW_LINE         ; print CR and LF on terminal
             JP INNER_LOOP         ; restart inner loop

;******************************************************************
;        INIT_UART                                                ;
;        Function....: Init serial port  8250, 16C450, OR 16C550  ;
;                      9600 Baud, 8 bit, 1 stopbit, 0 parity      ;
;        Output......:                                            ;
;        call........: PAUSE                      test 2 Feb 2007 ;
;******************************************************************
INIT_UART:   LD     A,$AA
             OUT    (UART7),A
             IN     A,(UART7)
             CP     $AA           ; TEST IF YOU COULD STORE AA
             JP     NZ,INITUART_FAIL  ; IF NOT, THE UART CAN'T BE FOUND
             LD     A,$55
             OUT    (UART7),A     ; 
             IN     A,(UART7)
             CP     $55           ; 
             JP     NZ,INITUART_FAIL
             LD     A,$01
             LD     (SER_ON),A
             JP     UART_OK

INITUART_FAIL:                      ; Handle if initialize UART fails
             LD     A,1
             LD     (UART_FAIL),A
             HALT


UART_OK:     LD     A,0
             LD     (UART_FAIL),A   ; UART OK FOUND
             LD     A,(SER_BAUD)
             CP     1
             JP     Z,UART1200
             CP     2
             JP     Z,UART2400
             CP     3
             JP     Z,UART4800
             CP     4
             JP     Z,UART9600
             CP     5
             JP     Z,UART19K2
             CP     6
             JP     Z,UART38K4
             CP     7
             JP     Z,UART57K6
             CP     8
             JP     Z,UART115K2
             ; IF NOTHING IS DEFINED 1200 WILL BE USED..


UART1200:    LD     A,80H
             OUT    (UART3),A     ; SET DLAB FLAG
             LD     A,96          ;  = 1,843,200 / ( 16 x 1200 )
             OUT    (UART0),A     ;
             LD     A,00H
             OUT    (UART1),A     ;
             LD     A,03H
             OUT    (UART3),A     ; Set 8 bit data, 1 stopbit

;             LD     A,03H
;             OUT    (UART4),A     ; Force DTR and RTS

             JP     INITRET       ; 0 parity, reset DLAP FLAG
UART2400:    LD     A,80H
             OUT    (UART3),A     ; SET DLAB FLAG
             LD     A,48          ; = 1,843,200 / ( 16 x 2400 )
             OUT    (UART0),A     ;
             LD     A,00H
             OUT    (UART1),A     ;
             LD     A,03H
             OUT    (UART3),A     ; Set 8 bit data, 1 stopbit
             JP     INITRET       ; 0 parity, reset DLAP FLAG
UART4800:    LD     A,80H
             OUT    (UART3),A     ; SET DLAB FLAG
             LD     A,24          ; = 1,843,200 / ( 16 x 4800 )
             OUT    (UART0),A     ;
             LD     A,00H
             OUT    (UART1),A     ;
             LD     A,03H
             OUT    (UART3),A     ; Set 8 bit data, 1 stopbit
             JP     INITRET       ; 0 parity, reset DLAP FLAG
UART9600:    LD     A,80H
             OUT    (UART3),A     ; SET DLAB FLAG
             LD     A,12          ; = 1,843,200 / ( 16 x 9600 )
             OUT    (UART0),A     ; Set BAUD rate til 9600
             LD     A,00H
             OUT    (UART1),A     ; Set BAUD rate til 9600
             LD     A,03H
             OUT    (UART3),A     ; Set 8 bit data, 1 stopbit
             JP     INITRET       ; 0 parity, reset DLAP FLAG
UART19K2:    LD     A,80H
             OUT    (UART3),A     ; SET DLAB FLAG
             LD     A,06          ; = 1,843,200 / ( 16 x 19,200 )
             OUT    (UART0),A     ;
             LD     A,0
             OUT    (UART1),A     ;
             LD     A,3
             OUT    (UART3),A     ; Set 8 bit data, 1 stopbit
             JP     INITRET       ; 0 parity, reset DLAP FLAG
UART38K4:    LD     A,80H
             OUT    (UART3),A     ; SET DLAB FLAG
             LD     A,03
             OUT    (UART0),A     ; = 1,843,200 / ( 16 x 38,400 )
             LD     A,00H
             OUT    (UART1),A     ;
             LD     A,03H
             OUT    (UART3),A     ; Set 8 bit data, 1 stopbit
             JP     INITRET       ; 0 parity, reset DLAP FLAG
UART57K6:    LD     A,80H
             OUT    (UART3),A     ; SET DLAB FLAG
             LD     A,02
             OUT    (UART0),A     ; = 1,843,200 / ( 16 x 57,600 )
             LD     A,00H
             OUT    (UART1),A     ;
             LD     A,03H
             OUT    (UART3),A     ; Set 8 bit data, 1 stopbit
             JP     INITRET       ; 0 parity, reset DLAP FLAG
UART115K2:    LD     A,80H
             OUT    (UART3),A     ; SET DLAB FLAG
             LD     A,01
             OUT    (UART0),A     ; = 1,843,200 / ( 16 x 115,200 )
             LD     A,00H
             OUT    (UART1),A     ;
             LD     A,03H
             OUT    (UART3),A     ; Set 8 bit data, 1 stopbit
                                  ; 0 parity, reset DLAP FLAG
INITRET:     RET





;******************************************************************
;        INIT_PIO                                                 ;
;        Function....: Init parallel port >82C55<                 ;
;******************************************************************
INIT_PIO:     
             LD     A,10011001B    ; A= IN, B= OUT C= IN
             OUT    (PIO1CONT),A
             RET





;******************************************************************
;        SUB-ROUTINE..: PAUSE                                      ;
;        Function....: Pause in 100uS. times value in BC          ;
;        Input.......: BC reg                                     ;
;        Output......: no                                         ;
;        call........: NONE                                       ;
;        Info........: KEA.      date: 2 Feb 2007                 ;
;******************************************************************
PAUSE:       PUSH   AF
             INC    B
             INC    C              ; ADJUST THE LOOP
PAUSELOOP1:  LD     A,13H          ; ADJUST THE TIME 13h IS FOR 4 MHZ
PAUSELOOP2:  DEC    A              ; DEC COUNTER. 4 T-states = 1 uS.
             JP     NZ,PAUSELOOP2  ; JUMP TO PAUSELOOP2 IF A <> 0.
             DEC    C              ; DEC COUNTER
             JP     NZ,PAUSELOOP1  ; JUMP TO PAUSELOOP1 IF C <> 0.

             DJNZ   PAUSELOOP1     ; JUMP TO PAUSELOOP1 IF B <> 0.
PAUSE_END:   POP    AF
             RET




;******************************************************************
;        TX_SER                                                   *
;        Function....: Send text data to serial port              *
;        Input.......: HL points at text start adr                *
;        Output......: Text to serialport                         *
;        uses........: A,HL                                       *
;        call........: TX_BUSY                    test 2 Feb 2007 *
;******************************************************************
TX_SER:      PUSH   AF
             LD     A,(SER_ON)     ; IF COM IS OFF
             CP     0              ; 
             JP     Z,TX_END
TX_SERLP:    LD     A,(HL)         ; GET CHARATER TO A
             CP     END            ; TEST FOR END BYTE
             JP     Z,TX_END       ; JUMP IF END BYTE IS FOUND
             CALL   TX_BUSY        ; WAIT FOR UART TO GET READY
             OUT    (UART0),A      ; THEN WRITE THE CHAR TO UART
             INC    HL             ; INC POINTER, TO NEXT CHAR
             JP     TX_SERLP       ; TRANSMIT LOOP
TX_END:      POP    AF
             RET

;******************************************************************
;        TX_SER_CHAR                                              *
;        Function....: Send character data to serial port         *
;        Input.......: register B contains character to send      *
;        Output......: character to serialport                    *
;        uses........: A, B                                       *
;        call........: TX_BUSY                    test 2 Feb 2007 *
;******************************************************************

TX_SER_CHAR:
             PUSH   AF
             LD     A,(SER_ON)     ; IF COM IS OFF
             CP     0              ; 
             JP     Z,TX_END_CHAR
             CALL   TX_BUSY        ; WAIT FOR UART TO GET READY
             LD     A,B
             OUT    (UART0),A      ; THEN WRITE THE CHAR TO UART
TX_END_CHAR:
             POP    AF
             RET


;******************************************************************
;        RX_SER_CHAR                                              *
;        Function....: Receive character data from serialport     *
;        Input.......: None                                       *
;        Output......: character from serialport in register B    *
;        uses........: A,B                                        *
;        call........: RX_BUSY                    test 2 Feb 2007 *
;******************************************************************
RX_SER_CHAR:
             PUSH   AF
             LD     A,(SER_ON)     ; IF COM IS OFF
             CP     0              ; 
             JP     Z,RX_END_CHAR
             CALL   RX_BUSY        ; WAIT FOR UART TO GET READY
             IN     A,(UART0)      ; THEN READ THE CHAR FROM THE UART
             LD     B,A            ; put received data character in B
                                   ; register and pass back to user
RX_END_CHAR:
             POP    AF
             RET



;******************************************************************
;        RX_BUSY                                                  *
;        Function....: WAIT FOR UART TO HAVE DATA IN BUFFER       *
;        Input.......: Bit 0 FROM UART MODEM CONTROL REGISTER     *
;******************************************************************
RX_BUSY:     PUSH   AF
RX_BUSYLP:   IN     A,(UART5)      ; READ Line Status Register
             BIT    0,A            ; TEST IF DATA IN RECEIVE BUFFER
             JP     Z,RX_BUSYLP    ; LOOP UNTIL DATA IS READY
             POP    AF
             RET


;******************************************************************
;        TX_BUSY                                                  *
;        Function....: WAIT FOR UART, TX BUFFER EMPTY             *
;        Input.......: Bit 5 FROM UART MODEM CONTROL REGISTER     *
;******************************************************************
TX_BUSY:     PUSH   AF
TX_BUSYLP:   IN     A,(UART5)     ; READ Line Status Register
             BIT    5,A           ; TEST IF UART IS READY TO SEND
             JP     Z,TX_BUSYLP   ; IF NOT REPEAT
             POP    AF
             RET



NEW_LINE:    LD     A,$0A         ; THIS GIVES A NEW LINE ON A TERMINAL
             CALL   TX_BUSY
             OUT    (UART0),A
             LD     A,$0D
             CALL   TX_BUSY
             OUT    (UART0),A
             RET




TXT_HELLO:  .BYTE " HELLO WORLD ",END




 ;  .include ctxt001.asm         ; YOU CAN INCLUDE OTHER ASM FILES AND USE-
                                 ; THE SUB ROUTINES FROM THEM.
 .text "\n\r  -END-OF-FILE-  \n\r"
 .end


