;___BOOT_LOADER________________________________________________________________________________________________________
;
;  
;  CREATED BY : 	ANDREW LYNCH 15-FEB-2007
;  MODIFIED BY : 	DAN WERNER 09 12.2009
;
;______________________________________________________________________________________________________________________


;__HARDWARE_INTERFACES________________________________________________________________________________________________________________ 
;
; PIO 82C55 I/O IS DECODED TO PORT 60-67
;
PORTA		 .EQU 	60H
PORTB		 .EQU 	61H
PORTC		 .EQU 	62H
PIOCONT 	 .EQU 	63H
;
; UART 16C450 SERIAL IS DECODED TO 68-6F
;
UART0:		 .EQU	68H		;   DATA IN/OUT
UART1:		 .EQU	69H		;   CHECK RX
UART2:		 .EQU	6AH		;   INTERRUPTS
UART3:		 .EQU	6BH		;   LINE CONTROL
UART4:		 .EQU	6CH		;   MODEM CONTROL
UART5:		 .EQU	6DH		;   LINE STATUS
UART6:		 .EQU	6EH		;   MODEM STATUS
UART7:		 .EQU	6FH		;   SCRATCH REG.
;
; MEMORY PAGE CONFIGURATION LATCH IS DECODED TO 78
;
MPCL:		 .EQU	78H		; CONTROL PORT, SHOULD ONLY BE CHANGED WHILE
;					  IN UPPER MEMORY PAGE 08000h-$FFFF OR LIKELY
;					  LOSS OF CPU MEMORY CONTEXT 
;
; MEMORY PAGE CONFIGURATION LATCH CONTROL PORT ( IO_Y3 ) INFORMATION
;
;	7 6 5 4  3 2 1 0      ONLY APPLICABLE TO THE LOWER MEMORY PAGE 00000h-$7FFF
;	^ ^ ^ ^  ^ ^ ^ ^
;	: : : :  : : : :--0 = A15 RAM/ROM ADDRESS LINE DEFAULT IS 0
;	: : : :  : : :----0 = A16 RAM/ROM ADDRESS LINE DEFAULT IS 0
;	: : : :  : :------0 = A17 RAM/ROM ADDRESS LINE DEFAULT IS 0
;	: : : :  :--------0 = A18 RAM/ROM ADDRESS LINE DEFAULT IS 0
;	: : : :-----------0 = A19 ROM ONLY ADDRESS LINE DEFAULT IS 0
;	: : :-------------0 = 
;	: :---------------0 = 
;	:-----------------0 = ROM SELECT (0=ROM, 1=RAM) DEFAULT IS 0
;
;
;IDE REGISTER		IO PORT		; FUNCTION
IDELO		 .EQU	020h		; DATA PORT (LOW BYTE)
IDEERR		 .EQU	021h		; READ: ERROR REGISTER; WRITE: PRECOMP
IDESECTC	 .EQU	022h		; SECTOR COUNT
IDESECTN	 .EQU	023h		; SECTOR NUMBER
IDECYLLO	 .EQU	024h		; CYLINDER LOW
IDECYLHI	 .EQU	025h		; CYLINDER HIGH
IDEHEAD		 .EQU	026h		; DRIVE/HEAD
IDESTTS		 .EQU	027h		; READ: STATUS; WRITE: COMMAND
IDEHI		 .EQU	028h		; DATA PORT (HIGH BYTE)
IDECTRL		 .EQU	02Eh		; READ: ALTERNATIVE STATUS; WRITE; DEVICE CONTROL
IDEADDR		 .EQU	02Fh		; DRIVE ADDRESS (READ ONLY)
;
FMSR:		 .EQU	036H		; ADDRESS OF MAIN STATUS REGISTER
FDATA:		 .EQU	037H		; FLOPPY DATA REGISTER
FLATCH:		 .EQU	03AH		; FLOPPY CONFIGURATION LATCH
FDMA:		 .EQU	03CH		; PSEUDO DMA ADDRESS

;
;
;__CONSTANTS_________________________________________________________________________________________________________________________ 
;	

RAMTOP:		 .EQU	0FFFFh	; HIGHEST ADDRESSABLE MEMORY LOCATION
MONSTART:	 .EQU	08000h	; START OF MONITOR
RAMBOTTOM:	 .EQU	08000h	; BEGINNING OF UPPER 32K RAM PAGE
END:		 .EQU	0FFh	; MARK END OF TEXT
CR:		 .EQU	0DH	; ASCII CARRIAGE RETURN CHARACTER
LF:		 .EQU	0AH	; ASCII LINE FEED CHARACTER
ESC:		 .EQU	1BH	; ASCII ESCAPE CHARACTER
BS:		 .EQU	08H	; ASCII BACKSPACE CHARACTER

CPMBOOT:	 .EQU    0E600H	; CPM COLD BOOT POINT
ROMSTART_MON:	 .EQU	03800h	; WHERE THE MONITOR IS STORED IN ROM
RAMTARG_MON:	 .EQU	08000h	; WHERE THE MONITOR STARTS IN RAM (ENTRY POINT)
MOVSIZ_MON:	 .EQU	01000h	; MONITOR IS 2KB IN LENGTH
TEMP_BUFFER	 .EQU	0B000H	; TEMP BUFFER
BOOT_BEGINL:	 .EQU	TEMP_BUFFER+506
BOOT_ENDL:	 .EQU	TEMP_BUFFER+508
BOOT_RUN:	 .EQU	TEMP_BUFFER+510


ROMSTART_CPM:	 .EQU	00A00h	; WHERE THE CCP+BDOS+BIOS IS STORED IN ROM
RAMTARG_CPM:	 .EQU	0D000h	; WHERE THE CCP+BDOS+BIOS STARTS IN RAM (ENTRY POINT)
MOVSIZ_CPM:	 .EQU	02A00h	; CCP, BDOS, + BIOS IS 6-7KB IN LENGTH
STACKTARG:	 .EQU	0CFFFh	; WHERE THE STACK POINTER STARTS IN RAM (ENTRY POINT)

MONITOR_ENTRY	 .EQU	0831Dh	; MONITOR ENTRY POINT (MAY CHANGE)
DMONITOR_ENTRY	 .EQU	08000h	; DSKY MONITOR ENTRY POINT (MAY CHANGE)

;
; FDC CONFIGURATION LATCH OUTPUT BIT PATTERNS
MOTOR:		 .EQU	000000000b	; BIT PATTERN IN LATCH FOR MOTOR CONTROL (ON)
TERMCN:		 .EQU	000000001b	; BIT PATTERN IN LATCH TO WRITE A TC STROBE
RESETL:		 .EQU	000000010b	; BIT PATTERN IN LATCH TO RESET ALL BITS
MINI:		 .EQU	000000100b	; BIT PATTERN IN LATCH TO SET MINI MODE FDC9229 LOW DENS=1, HIGH DENS=0
PRECOMP:	 .EQU	000100000b	; BIT PATTERN IN LATCH TO SET WRITE PRECOMP 125 NS:
FDDENSITY:	 .EQU	001000000b	; BIT PATTERN IN LATCH TO FLOPPY LOW DENSITY (HIGH IS 0)
FDREADY:	 .EQU	010000000b	; BIT PATTERN IN LATCH TO FLOPPY READY (P-34):

;______________________________________________________________________________________________________________________
;
; SELECT ONE OF THE FOLLOWING TO BE THE ROOT CONSOLE DEVICE, ONLY ONE CAN BE ACTIVE
;______________________________________________________________________________________________________________________
;
DSKY:		 .EQU	0		; 1= ACTIVE, 0=NOT ACTIVE
UART:		 .EQU	1		; 1= ACTIVE, 0=NOT ACTIVE
;______________________________________________________________________________________________________________________
	.ORG	0000H
	JP	MAIN

;______________________________________________________________________________________________________________________
;	MASKABLE INTERRUPT-PROGRAM
;	FUNCTION	:
;	INPUT		:
;	OUTPUT		: 
;	USES		: 
;	CALLS		: NONE
;	INFO		:
;	TESTED		: 2 FEB 2007
;______________________________________________________________________________________________________________________
	 .ORG	00038h		; INT MODE 1
	RETI			; RETURN FROM INTERRUPT

;______________________________________________________________________________________________________________________
;	NONMASKABLE INTERRUPT-PROGRAM
;	FUNCTION	:
;	INPUT		:
;	OUTPUT		: NONE
;	USES		: 
;	CALLS		: NONE
;	INFO		:
;	TESTED		: 2 FEB 2007
;______________________________________________________________________________________________________________________

	 .ORG	00066h		; HERE IS THE NMI ROUTINE
	RETI



;
;
;
;__MAIN_PROGRAM_____________________________________________________________________________________________________________________ 
;
	 .ORG	00100h		; FOR TESTING
MAIN:	 
	DI			; DISABLE INTERRUPT
	LD	SP,STACKTARG	; SET STACK POINTER TO TOP OFF RAM
	IM	1		; SET INTERRUPT MODE 1

	LD	HL,ROMSTART_MON	; WHERE IN ROM MONITOR IS STORED (FIRST BYTE)
	LD	DE,RAMTARG_MON	; WHERE IN RAM TO MOVE MONITOR TO (FIRST BYTE)
	LD	BC,MOVSIZ_MON	; NUMBER OF BYTES TO MOVE FROM ROM TO RAM
	LDIR			; PERFORM BLOCK COPY OF MONITOR TO UPPER RAM PAGE

	LD	HL,ROMSTART_CPM	; WHERE IN ROM CP/M IS STORED (FIRST BYTE)
	LD	DE,RAMTARG_CPM	; WHERE IN RAM TO MOVE MONITOR TO (FIRST BYTE)
	LD	BC,MOVSIZ_CPM	; NUMBER OF BYTES TO MOVE FROM ROM TO RAM
	LDIR			; PERFORM BLOCK COPY OF CP/M TO UPPER RAM PAGE

	EI			; ENABLE INTERRUPTS (ACCESS TO MONITOR WHILE CP/M RUNNING)
	CALL    INITSYSTEM	; INIT 

	JP	DOBOOT		; JUMP TO START OF MONITOR

;__DSKY_INIT________________________________________________________________________________________
;
;  SETUP 8255, MODE 0, PORT A=OUT, PORT B=IN, PORT C=OUT/OUT 
;     
;____________________________________________________________________________________________________
INITSYSTEM:
	.IF	DSKY
	LD	A, 82H			;
	OUT 	(PIOCONT),A		;
	LD	A,80H			;
	LD	(SEC),A			;
	LD	(SEC+1),A		;
	LD	(SEC+2),A		;
	LD	(SEC+3),A		;
	LD	(SEC+4),A		;
	LD	A,0CBH			;
	LD	(SEC+5),A		;
	LD	A,0CFH			;
	LD	(SEC+6),A		;
	LD	A,0D7H			;
	LD	(SEC+7),A		;
	.ENDIF
	LD	A,12			; SPECIFY BAUD RATE 9600 BPS (9600,8,NONE,1)
	LD	(SER_BAUD),A		; 
	CALL	INIT_UART		; INIT THE UART 
	RET				;
	
;__DOBOOT________________________________________________________________________________________________________________________ 
;
;	PERFORM BOOT FRONT PANEL ACTION
;________________________________________________________________________________________________________________________________
;
DOBOOT:
	 .IF DSKY

	LD	HL,BOOT			; POINT TO BOOT MESSAGE	
	CALL 	SEGDISPLAY		; DISPLAY MESSAGE
BOOTGETKEY:
	CALL	KB_GET			; GET KEY FROM KB INTO A
	CP	00H			; [0] PRESSED, BOOT DSKY MONITOR
	JP	Z,DMONITOR_ENTRY	; 
	CP	01H			; [1] PRESSED, BOOT SERIAL MONITOR
	JP	Z,MONITOR_ENTRY		; 	
	CP	02H			; [2] PRESSED, BOOT ROM CP/M
	JP	Z,CPMBOOT		; CP/M COLD BOOT ENTRY POINT GOCPM			;
	CP	03H			; [3] PRESSED, BOOT FLOPPY UNIT 0
	JP	Z,BOOTFLOPPYU0		; 	
	CP	04H			; [4] PRESSED, BOOT FLOPPY UNIT 1
	JP	Z,BOOTFLOPPYU1		; 	
	CP	05H			; [5] PRESSED, BOOT IDE PRIMARY
	JP	Z,BOOTIDEPRI		; 
	CP	06H			; [6] PRESSED, BOOT IDE SECONDARY
	JP	Z,BOOTIDESEC		;
	CP	07H			; [7] PRESSED, BOOT ATAPI PRIMARY
	JP	Z,BOOTATAPIPRI		; 
	CP	08H			; [8] PRESSED, BOOT ATAPI SECONDARY
	JP	Z,BOOTATAPISEC		;	
	LD	HL,BOOT			; POINT TO BOOT MESSAGE
	LD	A,00H			; BLANK OUT SELECTION,IT WAS INVALID
	LD	(HL),A			; STORE IT IN DISPLAY BUFFER
	CALL	SEGDISPLAY		; DISPLAY THE BUFFER
	JR	BOOTGETKEY		; NO VALID KEY, LOOP
	 .ENDIF
	 .IF UART
	LD	HL,MSG_START		;POINT AT TEXT
	CALL	MSG			;SHOW WE'RE HERE
	CALL	CRLFA			; CR,LF,>
	LD	HL,KEYBUF		; SET POINTER TO KEYBUF AREA
	CALL 	GETLN			; GET A LINE OF INPUT FROM THE USER
	LD	HL,KEYBUF		; RESET POINTER TO START OF KEYBUF
        LD      A,(HL)			; LOAD FIRST CHAR INTO A (THIS SHOULD BE THE COMMAND)
	INC	HL			; INC POINTER
	
	CP	'0'			; [0] PRESSED, BOOT DSKY MONITOR
	JP	Z,DMONITOR_ENTRY	; 
	CP	'1'			; [1] PRESSED, BOOT SERIAL MONITOR
	JP	Z,MONITOR_ENTRY		; 	
	CP	'2'			; [2] PRESSED, BOOT ROM CP/M
	JP	Z,CPMBOOT		; CP/M COLD BOOT ENTRY POINT GOCPM			;
	CP	'3'			; [3] PRESSED, BOOT FLOPPY UNIT 0
	JP	Z,BOOTFLOPPYU0		; 	
	CP	'4'			; [4] PRESSED, BOOT FLOPPY UNIT 1
	JP	Z,BOOTFLOPPYU1		; 	
	CP	'5'			; [5] PRESSED, BOOT IDE PRIMARY
	JP	Z,BOOTIDEPRI		; 
	CP	'6'			; [6] PRESSED, BOOT IDE SECONDARY
	JP	Z,BOOTIDESEC		;
	CP	'7'			; [7] PRESSED, BOOT ATAPI PRIMARY
	JP	Z,BOOTATAPIPRI		; 
	CP	'8'			; [8] PRESSED, BOOT ATAPI SECONDARY
	JP	Z,BOOTATAPISEC		;	
	JR	DOBOOT			; NO VALID KEY, LOOP	
	 .ENDIF
	
	
BOOTFLOPPYU0:
	LD	A,00			; SET UNIT ID
	LD	(UNIT),A		; STORE
	JP	BOOTFLOPPY		;
BOOTFLOPPYU1:
	LD	A,01			; SET UNIT ID
	LD	(UNIT),A		; STORE
BOOTFLOPPY:
	CALL	SETUPDRIVE	; SETUP FLOPPY PARAMETERS
	LD	A,00			; 
	LD	(PTRACK),A		; SET TRACK TO 0
	LD	(HEAD),A		; SET HEAD TO 0
	LD	A,03			;
	LD	(SECTOR),A		; SET SECTOR TO 3
	LD	HL,TEMP_BUFFER		; SET TEMP BUFFER
	LD	(TEMPW),HL		;
	CALL	READ_FLOPPY_SECTOR	; READ METADATA
	LD	A,00			; 
	LD	(PTRACK),A		; SET TRACK TO 0
	LD	(HEAD),A		; SET HEAD TO 0
	LD	A,04			;
	LD	(SECTOR),A		; SET SECTOR TO 1
	LD	HL,(BOOT_BEGINL)	; SET TOP OF CP/M
	LD	(TEMPW),HL		;
BOOTFLOPPYLOOP:
	CALL	READ_FLOPPY_SECTOR	; READ A SECTOR
	LD	HL,(TEMPW)		;
	LD	BC,0200H		;
	ADD	HL,BC			;
	LD	(TEMPW),HL		;
	LD	A,H			;
	CP	00H			;
	JP	Z,BOOTFLOPPYEXIT	;
	PUSH    AF			;
	LD      A,(BOOT_ENDL+1)		;
	PUSH	AF			;
	POP	DE			;
	POP	AF			;
	CP	D			;
	JP	P,BOOTFLOPPYEXIT	;
	LD	A,(SECTOR)		;	
	INC	A			; POINT TO NEXT SECTOR
	LD	(SECTOR),A		;
	CP	0AH			;
	JP	NZ,BOOTFLOPPYLOOP	; LOOP
	LD	A,01			;
	LD	(SECTOR),A		; SET SECTOR TO 1
	LD	A,(HEAD)		;
	INC	A			;
	LD	(HEAD),A		;
	CP	02H			;
	JP	NZ,BOOTFLOPPYLOOP	; LOOP
	LD	A,00			;
	LD	(HEAD),A		; SET HEAD TO 1
	LD	A,(PTRACK)		;
	INC	A			;
	LD	(PTRACK),A		;
	CP	02H			;
	JP	NZ,BOOTFLOPPYLOOP	; LOOP
BOOTFLOPPYEXIT:	
	LD	HL,(BOOT_RUN)		;
	JP	(HL)			; GO TO CP/M
BOOTATAPIPRI:
	LD	A,00H
	LD	(IDEDEVICE),A		; SET PRIMARY DEVICE
	JR	BOOTATAPI		;
BOOTATAPISEC:
	LD	A,10H			;
	LD	(IDEDEVICE),A		; SET SECONDARY DEVICE
BOOTATAPI:
	CALL	IDE_SOFT_RESET		;
	CALL	ATAPI_DEVICE_SELECTION	;
	CALL	DELAY12			;
	CALL 	REQUEST_SENSE_LOOP	;
	 .IF DSKY
	LD	HL,SEC			; DISPLAY PROMPT
	CALL	SEGDISPLAY		; 
	 .ENDIF
	LD	DE,0000H		; SET SECTOR HIGH WORD TO 00H
	LD	HL,0002H		; SET SECTOR LOW WORD TO 03H
	LD	IX,TEMP_BUFFER		; SET TOP OF CP/M
	 .IF DSKY	 
	CALL	DSPSECTOR		;
	 .ENDIF
	CALL	ATAPI_READ_SECTOR	; READ METADATA
	LD	DE,0000H		; SET SECTOR HIGH WORD TO 00H
	LD	HL,0003H		; SET SECTOR LOW WORD TO 03H
	LD	IX,(BOOT_BEGINL)	; SET TOP OF CP/M
BOOTATAPILOOP:
	 .IF DSKY	 
	CALL	DSPSECTOR		;
	 .ENDIF
	CALL	ATAPI_READ_SECTOR	; READ A SECTOR
	INC	HL			; POINT TO NEXT SECTOR
	PUSH	HL			; STORE HL
	LD	A,(BOOT_ENDL+1)		;
	LD	B,A			;
	PUSH	IX			;
	POP	HL			; ADDRESS -> HL
	LD	A,H			;
	POP	HL			;
	CP	00H			;
	JP	Z,BOOTATAPIEXIT		;
	CP	B			;
	JP	P,BOOTATAPIEXIT		;
	JP	BOOTATAPILOOP		;
BOOTATAPIEXIT:	
	LD	HL,(BOOT_RUN)		;
	JP	(HL)			; GO TO CP/M
BOOTIDEPRI:
	LD	A,00H
	LD	(IDEDEVICE),A		; SET PRIMARY DEVICE
	JR	BOOTIDE			;
BOOTIDESEC:
	LD	A,10H			;
	LD	(IDEDEVICE),A		; SET SECONDARY DEVICE
BOOTIDE:
	CALL	IDE_SOFT_RESET		;
	 .IF DSKY
	LD	HL,SEC			; DISPLAY PROMPT
	CALL	SEGDISPLAY		; 
	 .ENDIF
 	LD	DE,0000H		; SET SECTOR HIGH WORD TO 00H
	LD	HL,0002H		; SET SECTOR LOW WORD TO 03H
	LD	IX,TEMP_BUFFER		; SET TOP OF CP/M
	 .IF DSKY	 
	CALL	DSPSECTOR		;
	 .ENDIF
	CALL	IDE_READ_SECTOR		; READ METADATA
	LD	DE,0000H		; SET SECTOR HIGH WORD TO 00H
	LD	HL,0003H		; SET SECTOR LOW WORD TO 03H
	LD	IX,(BOOT_BEGINL)	; SET TOP OF CP/M
BOOTIDELOOP:
	PUSH	BC			;
	PUSH	HL			;
	PUSH	DE			;
	 .IF DSKY
	CALL	DSPSECTOR		;
	 .ENDIF
	CALL	IDE_READ_SECTOR		; READ A SECTOR
	POP	DE			;
	POP	HL			;
	POP	BC			;
	INC	HL			; POINT TO NEXT SECTOR
	PUSH	HL			; STORE HL
	LD	A,(BOOT_ENDL+1)		;
	LD	B,A			;
	PUSH	IX
	POP	HL			; ADDRESS -> HL
	LD	A,H			;
	POP	HL			;
	CP	00H			;
	JP	Z,BOOTIDEEXIT		;
	CP	B			;
	JP	P,BOOTIDEEXIT		;
	JP	BOOTIDELOOP		;
BOOTIDEEXIT:	
	LD	HL,(BOOT_RUN)		;
	JP	(HL)			; GO TO CP/M
	
	
	 .IF DSKY
;__KB_GET____________________________________________________________________________________________
;
;  GET A SINGLE KEY AND DECODE
;     
;____________________________________________________________________________________________________
KB_GET:
	PUSH 	HL			; STORE HL
KB_GET_LOOP:				; WAIT FOR KEY
	CALL	KB_SCAN			;  SCAN KB ONCE
	CP	00H			;  NULL?
	JR	Z,KB_GET_LOOP		;  LOOP WHILE NOT ZERO
	LD      D,A			;  STORE A
	LD	A,4FH			;  SCAN ALL COL LINES
	OUT 	(PORTC),A		;  SEND TO COLUMN LINES
        CALL    DELAY12			;  DELAY TO ALLOW LINES TO STABILIZE
KB_CLEAR_LOOP:				; WAIT FOR KEY TO CLEAR
	IN	A,(PORTB)		;  GET ROWS
	CP	00H 			;  ANYTHING PRESSED?
	JR	NZ,KB_CLEAR_LOOP	;  YES, EXIT 
	LD	A,D			;  RESTORE A
	LD	D,00H			;
	LD	HL,KB_DECODE		;  POINT TO BEGINNING OF TABLE	
KB_GET_LLOOP:
	CP	(HL)			;  MATCH?	
	JR	Z,KB_GET_DONE		;  FOUND, DONE
	INC	HL
	INC	D			;  D + 1	
	JP	NZ,KB_GET_LLOOP		;  NOT FOUND, LOOP UNTIL EOT			
KB_GET_DONE:
	LD	A,D			;  RESULT INTO A
	POP	HL			; RESTORE HL
	RET



;__KB_SCAN____________________________________________________________________________________________
;
;  SCAN KEYBOARD MATRIX FOR AN INPUT
;     
;____________________________________________________________________________________________________
KB_SCAN:

	LD      C,0000H
	LD	A,41H			;  SCAN COL ONE
	OUT 	(PORTC),A		;  SEND TO COLUMN LINES
        CALL    DELAY12			;  DELAY TO ALLOW LINES TO STABILIZE
	IN	A,(PORTB)		;  GET ROWS
	CP	00H 			;  ANYTHING PRESSED?
	JR	NZ,KB_SCAN_FOUND	;  YES, EXIT 

	LD      C,0040H
	LD	A,42H			;  SCAN COL TWO
	OUT 	(PORTC),A		;  SEND TO COLUMN LINES
        CALL    DELAY12			;  DELAY TO ALLOW LINES TO STABILIZE
	IN	A,(PORTB)		;  GET ROWS
	CP	00H 			;  ANYTHING PRESSED?
	JR	NZ,KB_SCAN_FOUND	;  YES, EXIT 

	LD      C,0080H
	LD	A,44H			;  SCAN COL THREE
	OUT 	(PORTC),A		;  SEND TO COLUMN LINES
        CALL    DELAY12			;  DELAY TO ALLOW LINES TO STABILIZE
	IN	A,(PORTB)		;  GET ROWS
	CP	00H 			;  ANYTHING PRESSED?
	JR	NZ,KB_SCAN_FOUND	;  YES, EXIT 

	LD      C,00C0H			;
	LD	A,48H			;  SCAN COL FOUR
	OUT 	(PORTC),A		;  SEND TO COLUMN LINES
        CALL    DELAY12			;  DELAY TO ALLOW LINES TO STABILIZE
	IN	A,(PORTB)		;  GET ROWS
	CP	00H 			;  ANYTHING PRESSED?
	JR	NZ,KB_SCAN_FOUND	;  YES, EXIT 

	LD	A, 40H			;  TURN OFF ALL COLUMNS
	OUT 	(PORTC),A		;  SEND TO COLUMN LINES
	LD	A, 00H			;  RETURN NULL
	RET				;  EXIT

KB_SCAN_FOUND:
	AND	3FH			;  CLEAR TOP TWO BITS
	OR	C			;  ADD IN ROW BITS 
	LD	C,A			;  STORE VALUE
	LD	A, 00H			;  TURN OFF ALL COLUMNS
	OUT 	(PORTC),A		;  SEND TO COLUMN LINES
	LD	A,C			;  RESTORE VALUE
	RET


;__DECODEDISPLAY_____________________________________________________________________________________
;
;  DISPLAY CONTENTS OF DISPLAYBUF IN DECODED HEX BITS 0-3 ARE DISPLAYED DIG, BIT 7 IS DP
;     
;____________________________________________________________________________________________________
DECODEDISPLAY:
	PUSH	BC			; STORE BC
	PUSH	HL			; STORE HL
	LD	HL,SEGDECODE		; POINT HL TO DECODE TABLE
	LD	B,00H			; RESET HIGH BYTE
	LD	C,A			; CHAR INTO LOW BYTE
	ADD	HL,BC			; SET TABLE POINTER
	LD	A,(HL)			; GET VALUE
	POP	HL			; RESTORE HL
	POP	BC			; RESTORE BC
	RET


	
	
;__FLSEGDISPLAY________________________________________________________________________________________
;
;  DISPLAY CONTENTS OF TRACK, SECTOR, ST0, ST1 ON DSKY
;     
;____________________________________________________________________________________________________
FLSEGDISPLAY:
	PUSH	AF			; STORE AF
	PUSH	BC			; STORE BC
	PUSH	HL			;
	LD	A,(PTRACK)		;
	AND	0FH			;
	LD	(DISPLAYBUF+6),A	;
	LD	A,(PTRACK)		;
	AND	0F0H			;
	SRL	A			;
	SRL	A			;
	SRL	A			;
	SRL	A			;
	LD	(DISPLAYBUF+7),A	;			
	LD	A,(SECTOR)		;
	AND	0FH			;
	LD	(DISPLAYBUF+4),A	;
	LD	A,(SECTOR)		;
	AND	0F0H			;
	SRL	A			;
	SRL	A			;
	SRL	A			;
	SRL	A			;
	LD	(DISPLAYBUF+5),A	;
	LD	A,(ST0)			;
	AND	0FH			;
	LD	(DISPLAYBUF+2),A	;
	LD	A,(ST0)			;
	AND	0F0H			;
	SRL	A			;
	SRL	A			;
	SRL	A			;
	SRL	A			;
	LD	(DISPLAYBUF+3),A	;			
	LD	A,(ST1)			;
	AND	0FH			;
	LD	(DISPLAYBUF),A		;
	LD	A,(ST1)			;
	AND	0F0H			;
	SRL	A			;
	SRL	A			;
	SRL	A			;
	SRL	A			;
	LD	(DISPLAYBUF+1),A	;
	LD	HL,DISPLAYBUF		;
	LD	BC,0007H		;
	ADD	HL,BC			;
	LD	B,08H			; SET DIGIT COUNT
	LD	A,40H			; SET CONTROL PORT 7218 TO OFF
	OUT	(PORTC),A		; OUTPUT
	CALL 	DELAY12			; WAIT
	LD	A,0D0H			; SET CONTROL TO 1111 (DATA COMING, HEX DECODE, DECODE, NORMAL)
	JP	SEGDISPLAY1		;

;__SEGDISPLAY________________________________________________________________________________________
;
;  DISPLAY CONTENTS OF DISPLAYBUF IN DECODED HEX BITS 0-3 ARE DISPLAYED DIG, BIT 7 IS DP
;     
;____________________________________________________________________________________________________
SEGDISPLAY:
	PUSH	AF			; STORE AF
	PUSH	BC			; STORE BC
	PUSH	HL			; STORE HL
	LD	BC,0007H	
	ADD	HL,BC
	LD	B,08H			; SET DIGIT COUNT
	LD	A,40H			; SET CONTROL PORT 7218 TO OFF
	OUT	(PORTC),A		; OUTPUT
	CALL 	DELAY12			; WAIT
	LD	A,0F0H			; SET CONTROL TO 1111 (DATA COMING, HEX DECODE,NO DECODE, NORMAL)
SEGDISPLAY1:				;

	OUT	(PORTA),A		; OUTPUT TO PORT
	LD	A,80H			; STROBE WRITE PULSE WITH CONTROL=1
	OUT	(PORTC),A		; OUTPUT TO PORT
	CALL 	DELAY12			; WAIT
	LD	A,40H			; SET CONTROL PORT 7218 TO OFF
	OUT	(PORTC),A		; OUTPUT
SEGDISPLAY_LP:		
	LD	A,(HL)			; GET DISPLAY DIGIT
	OUT	(PORTA),A		; OUT TO PORTA
	LD	A,00H			; SET WRITE STROBE
	OUT	(PORTC),A		; OUT TO PORTC
	CALL	DELAY12			; DELAY
	LD	A,40H			; SET CONTROL PORT OFF
	OUT	(PORTC),A		; OUT TO PORTC
	CALL	DELAY12			; WAIT
	DEC	HL			; INC POINTER
	DJNZ	SEGDISPLAY_LP		; LOOP FOR NEXT DIGIT
	POP	HL			; RESTORE HL
	POP	BC			; RESTORE BC
	POP	AF			; RESTORE AF
	RET
	 .ENDIF

;__DELAY24__________________________________________________________________________________________________________________________ 
;
; 	DELAY 24US
;________________________________________________________________________________________________________________________________
;
	
DELAY24:	
					; JP= 10T	
	PUSH	IX			; 15T
	POP	IX			; 14T
	PUSH	IX			; 15T
	POP	IX			; 14T
DELAY12:
	PUSH	IX			; 15T
	POP	IX			; 14T
	RET				; 10T

	
	
;________________________________________________________________________________________________________________________________ 
;
;   ATAPI CODE
;________________________________________________________________________________________________________________________________ 

;__ATAPI_READ_SECTOR_____________________________________________________________________________________________________________ 
;  READ ATAPI SECTOR   
;
;   D E H L = SECTOR (DOUBLE WORD) TO READ 
;   IX = READ TO ADDRESS 
;
;   ON RETURN IX WILL BE POINTED TO NEXT BYTE AFTER READ OPERATION 
;________________________________________________________________________________________________________________________________ 
ATAPI_READ_SECTOR:
	PUSH	BC			; STORE BC
	PUSH	DE			; STORE DE
	PUSH	HL			; STORE HL
	LD	A,0A8H			; 
	LD	(READ_DISK_PACKET+0),A	;		 
	LD	A,00H			; 
	LD	(READ_DISK_PACKET+1),A	;
	LD	A,D			; STORE SECTOR ADDRESS TO PACKET
	LD	(READ_DISK_PACKET+2),A	;
	LD	A,E			;
	LD	(READ_DISK_PACKET+3),A	;
	LD	A,H			;
	LD	(READ_DISK_PACKET+4),A	;
	LD	A,L			;
	LD	(READ_DISK_PACKET+5),A	;
	LD	A,00H			; 
	LD	(READ_DISK_PACKET+6),A	;
 	LD	A,00H			; 
	LD	(READ_DISK_PACKET+7),A	;
 	LD	A,00H			; 
	LD	(READ_DISK_PACKET+8),A	;
 	LD	A,01H			; 
	LD	(READ_DISK_PACKET+9),A	;
 	LD	A,00H			; 
	LD	(READ_DISK_PACKET+10),A	;
 	LD	A,00H			; 
	LD	(READ_DISK_PACKET+11),A	;

	CALL	REQUEST_SENSE_LOOP	; GET ATAPI SENSE CODES TO CLEAR ERRORS
	LD	HL,READ_DISK_PACKET	; SET POINTER TO READ SECTOR PACKET
	CALL	ATAPI_SEND_PACKET	; SEND PACKET COMMAND
	CALL	ATAPI_READ_DATA		; 
	POP	HL			; RESTORE HL
	POP	DE			; RESTORE DE
	POP	BC			; RESTORE BC
	RET				;



;__ATAPI_SEND_PACKET_____________________________________________________________________________________________________________ 
;  SEND PACKET POINTED TO BY HL TO ATAPI DRIVE   
;
;________________________________________________________________________________________________________________________________ 
ATAPI_SEND_PACKET:

	CALL	IDE_WAIT_BUSY_READY	; MAKE SURE DRIVE IS READY TO PROCEED
	CALL	IDE_WAIT_DRQ_ZERO	;
					;
	LD	A,0AH			;
	OUT	(IDECTRL),A		; DISABLE INT
	LD	A,00H			;
	OUT	(IDEERR),A		;
	LD	A,00H			;
	OUT	(IDESECTC),A		; 
	LD	A,00H			;
	OUT	(IDESECTN),A		;
	LD	A,00H			;
	OUT	(IDECYLLO),A		; 
	LD	A,60H			;
	OUT	(IDECYLHI),A		; 
	LD	A,(IDEDEVICE)		;
	OUT	(IDEHEAD),A		; BIT 4 SELECTS DEVICE
	LD	A,0A0H			;
	OUT	(IDESTTS),A		;
					;
	CALL	IDE_WAIT_DRQ_READY	; MAKE SURE DRIVE IS READY TO PROCEED
					;
	LD	B,6			; SEND 12 BYTES (6 WORDS)
					;
ATAPI_SEND_PACKET_LOOP:
	LD	A,(HL)			; GET BYTE
	LD	D,A			; STORE LOW BYTE IN D
	INC	HL			; INC POINTER
	LD	A,(HL)			; GET HIGH BYTE
	OUT	(IDEHI),A		; STORE HIGH BYTE
	LD	A,D			; MOVE LOW BYTE INTO A
	OUT	(IDELO),A		; STORE LOW BYTE
	INC	HL			; INC POINTER
	IN	A,(IDECTRL)		; GET STATUS
	DJNZ	ATAPI_SEND_PACKET_LOOP	; LOOP
					;
	CALL	IDE_WAIT_BUSY_READY	; MAKE SURE DRIVE IS READY TO PROCEED
	IN	A,(IDECTRL)		; READ STATUS (FOR DELAY)
					;
	RET				;




;__ATAPI_READ_DATA__________________________________________________________________________________________________________
;      READ DATA BLOCK INTO SECTOR_BUFFER
;
; IX = POINTER TO BUFFER
;
;___________________________________________________________________________________________________________________________

ATAPI_READ_DATA:
	LD	B,0			; 256 WORDS (512 BYTES PER SECTOR)
	CALL	IDE_WAIT_BUSY_READY	; MAKE SURE DRIVE IS READY TO PROCEED
	IN	A,(IDESTTS)		; READ  REG
	AND	000001000b		; MASK OFF BIT
	CP	08H			; IS DATA WAITING?
	JR	NZ,ATAPI_READ_DATA_EXIT	; NO, EXIT
ATAPI_READ_DATA_LOOP:
	IN	A,(IDELO)		;
	LD	(IX+0),A		;
	INC	IX			;
	IN	A,(IDEHI)		;
	LD	(IX+0),A		;
	INC	IX			;
	DJNZ	ATAPI_READ_DATA_LOOP	;
ATAPI_READ_DATA_EXIT:
	RET				;

;__IDE_SOFT_RESET__________________________________________________________________________________________________________
;      RESET IDE DEVICE
;___________________________________________________________________________________________________________________________
IDE_SOFT_RESET:

	LD	A,000001110b		;NO INTERRUPTS, RESET DRIVE = 1
	OUT	(IDECTRL),A		;
	CALL	DELAY12			;
	LD	A,000001010b		;NO INTERRUPTS, RESET DRIVE = 0
	OUT	(IDECTRL),A		;
	CALL	IDE_WAIT_BUSY_READY	;
	RET

;___REQUEST_SENSE_LOOP____________________________________________________________________________________
;
;	ATAPI_REQUEST SENSE DATA
;_________________________________________________________________________________________________________			
REQUEST_SENSE_LOOP:
	PUSH	HL			;
	PUSH	IX			;
	PUSH	AF			;
	LD	HL,ATAPI_REQUEST_SENSE	;
	CALL	ATAPI_SEND_PACKET	;
	CALL	ATAPI_WAIT_BUSY_READY	; MAKE SURE DRIVE IS READY TO PROCEED
	JP	NC,REQUEST_SENSE_EXIT	; ERROR, RETURN
	LD	B,0			; 256 WORDS (512 BYTES PER SECTOR)
REQUEST_SENSE_LOOP1:			;
	IN	A,(IDELO)		;
	INC	IX			;
	IN	A,(IDEHI)		;
	INC	IX			;
	DJNZ	REQUEST_SENSE_LOOP1	;
	RRD				; DELAY ONLY
	IN	A,(IDESTTS)		;READ ERROR REG
	AND	000000001b		;MASK OFF BIT
	JR	NZ,REQUEST_SENSE_LOOP	;
REQUEST_SENSE_EXIT:
	POP	AF			;
	POP	IX			;
	POP	HL			;
	RET

;___ATAPI_WAIT_BUSY_READY________________________________________________________________________________
;
;	WAIT FOR ATAPI CHANNEL TO BE READY
;________________________________________________________________________________________________________			
ATAPI_WAIT_BUSY_READY:
	LD	DE,0			; CLEAR OUT DE
ATAPI_WBSY:				;
	LD	B,5			; SETUP TIMEOUT
ATAPI_DLP:				;
	DJNZ	ATAPI_DLP		; 
	INC	DE			;
	LD	A,D			;
	OR	E			;
	JR	Z,ATAPI_TO		;
	IN	A,(IDESTTS)		; READ ERROR REG
	AND	010000000b		; MASK OFF BUSY BIT
	JR	NZ,ATAPI_WBSY		; WE WANT BUSY(7) TO BE 0 
	SCF				; CARRY 1 = OK
	RET				;
ATAPI_TO:				;
	XOR	A			; CARRY 0 = TIMED OUT
	RET				;



;__ATAPI_DEVICE_SELECTION___________________________________________________________________________________________________
; IDE SELECT DEVICE 
;___________________________________________________________________________________________________________________________
ATAPI_DEVICE_SELECTION:

	CALL	IDE_WAIT_BUSY_READY	;	
	LD	A,(IDEDEVICE)		; SELECTS DEVICE
	OR	0A0H			;
	OUT	(IDEHEAD),A		;	
	CALL	IDE_WAIT_BUSY_READY	;
	RET				;


;__IDE_WAIT_BUSY_READY______________________________________________________________________________________________________
; WAIT FOR BUSY TO BE READY
;___________________________________________________________________________________________________________________________
IDE_WAIT_BUSY_READY:
	PUSH	DE			;
	PUSH 	BC			;
	LD	DE,0			;
IDE_WBSY:
	LD	B,5			;
IDE_DLP:
	DJNZ	IDE_DLP			;
	INC	DE			;
	LD	A,D			;
	OR	E			;
	JR	Z,IDE_TO		;
	IN	A,(IDESTTS)		; READ ERROR REG
	AND	010000000b		; MASK OFF BUSY BIT
	JR	NZ,IDE_WBSY		; WE WANT BUSY(7) TO BE 0 
	SCF				; CARRY 1 = OK
	POP	BC			;
	POP	DE			;
	RET				;
IDE_TO:
	POP	BC			;
	POP	DE			;
	XOR	A			; CARRY 0 = TIMED OUT
	RET				;
	

;__IDE_WAIT_DRDY_READY______________________________________________________________________________________________________
; WAIT FOR DRDY
;___________________________________________________________________________________________________________________________
IDE_WAIT_DRDY_READY:
	IN	A,(IDESTTS)		; READ ERROR REG
	AND	001000000b		; MASK OFF RDY BIT
	JR	Z,IDE_WAIT_DRDY_READY	; WE WANT RDY(6) TO BE 1 
	RET

;__IDE_WAIT_DRQ_READY_______________________________________________________________________________________________________
; WAIT FOR DRQ 
;___________________________________________________________________________________________________________________________
IDE_WAIT_DRQ_READY:
	IN	A,(IDESTTS)		; READ ERROR REG
	AND	000001000b		; MASK OFF RDY BIT
	JR	Z,IDE_WAIT_DRQ_READY	; WE WANT DRQ(3) TO BE 1
	RET

;__IDE_WAIT_DRQ_ZERO________________________________________________________________________________________________________
; WAIT FOR DRQ = 0
;___________________________________________________________________________________________________________________________
IDE_WAIT_DRQ_ZERO:
	IN	A,(IDESTTS)		; READ ERROR REG
	AND	000001000b		; MASK OFF RDY BIT
	JR	NZ,IDE_WAIT_DRQ_ZERO	; WE WANT DRQ(3) TO BE 0
	RET
	
;___READ_FLOPPY_SECTOR_____________________________________________________________________________________
;
;	READ A SECTOR FROM A FLOPPY DISK
;
;________________________________________________________________________________________________________
READ_FLOPPY_SECTOR:
 	 .IF	DSKY
	CALL	FLSEGDISPLAY		;
	 .ENDIF
	CALL	FLOPPYREAD		; READ THE FLOPPY DISK SECTOR
	LD	A,(ST0)			; GET STATUS FLAG 0
	AND	0F8H			; MASK OF DRIVE AND HEAD SELECTION
	LD	B,A			; MOVE STATUS FLAG 0 TO B
	LD	A,(ST1)			; GET STATUS FLAG 1
	OR	B			; IF ZERO READ WAS OK
	JP	NZ,READ_FLOPPY_SECTOR	;
	
	 .IF	DSKY
	CALL	FLSEGDISPLAY		;
	 .ENDIF
	RET
	
;__SETUPDRIVE__________________________________________________________________________________________________________________________ 
;
;	SETUP FLOPPY DRIVE SETTINGS 
;________________________________________________________________________________________________________________________________
;
;
;
SETUPDRIVE:
	LD	A,RESETL		; RESET SETTINGS
	OR	MINI			; SELECT MINI FLOPPY (LOW DENS=1, HIGH DENS=0)
	OR	PRECOMP			; SELECT PRECOMP 
	OR	FDDENSITY		; SELECT DENSITY
	OR	FDREADY			; SELECT READY SIGNAL
	LD	(FLATCH_STORE),A	; SAVE SETTINGS
	LD	A,01H			;
	LD	(UNIT),A		; SET UNIT 1
	LD	A,2			; DENSITY
	LD	(DENS),A		;
	LD	A,09			;
	LD	(EOTSEC),A		; LAST SECTOR OF TRACK			
	LD	A,7FH			;
	LD	(SRTHUT),A		; STEP RATE AND HEAD UNLOAD TIME
	LD	A,05H			;
	LD	(HLT),A			; HEAD LOAD TIME
	LD	A,0DH			;
	LD	(GAP),A			; GAP 
	LD	A,80H			;
	LD	(SECSIZ),A		; SECTOR SIZE /4
					;
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
					;					
	LD	HL,FLATCH_STORE		; POINT TO FLATCH
	RES	1,(HL)			; SET MOTOR ON
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
	NOP				;
	NOP				;
	LD	A,00H			; ZERO TRACK
	LD	(PTRACK),A		; STORE TRACK
	CALL	SETTRACK		; DO IT	
	NOP				;
	NOP				;
	LD	HL,FLATCH_STORE		; POINT TO FLATCH
	SET	1,(HL)			; SET MOTOR OFF
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
	RET
;
;__OUTFLATCH__________________________________________________________________________________________________________________________ 
;
;	SEND SETTINGS TO FLOPPY CONTROLLER
;________________________________________________________________________________________________________________________________
;
OUTFLATCH:
	LD	A,(FLATCH_STORE)	; SET A TO SETTINGS
	OUT	(FLATCH),A		; OUTPUT TO CONTROLLER
	RET

		
;__FLOPPYREAD__________________________________________________________________________________________________________________________ 
;
; 	READ A FLOPPY DISK SECTOR 	
;________________________________________________________________________________________________________________________________
;	
FLOPPYREAD:
	LD	A,46H			; BIT 6 SETS MFM, 06H IS READ COMMAND
	LD	(CMD),A			;
	LD	HL,FLATCH_STORE		; POINT TO FLATCH
	SET	1,(HL)			; SET MOTOR OFF
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
					;
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CP	0FFh			; DID IT RETURN WITH ERROR CODE?
	JP	Z,DSKEXIT		; IF YES, EXIT WITH ERROR CODE
					;	
	LD	A,(UNIT)		; GET DISK UNIT NUMBER
	AND	03H			; MASK FOR FOUR DRIVES 
	LD	B,A			; PARK IT IN B
	LD	A,(HEAD)		; GET HEAD SELECTION
	AND	01H			; INSURE SINGLE BIT
	RLA				;
	RLA				; MOVE HEAD TO BIT 2 POSITION
	OR	B			; OR HEAD TO UNIT BYTE IN COMMAND BLOCK
	LD	(UNIT),A		; STORE IN UNIT
					;
	LD	HL,FLATCH_STORE		; POINT TO FLATCH
	RES	1,(HL)			; SET MOTOR ON
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER	
					;
	LD	A,003h			; SPECIFY COMMAND
	CALL	PFDATA			; PUSH IT
	LD	A,(SRTHUT)		; STEP RATE AND HEAD UNLOAD TIME
	CALL	PFDATA			; PUSH THAT
	LD	A,(HLT)			;
	CALL	PFDATA			; PUSH THAT
					;
	CALL	SETTRACK		; PERFORM SEEK TO TRACK
					;
	JP	NZ,DSKEXIT		; IF ERROR, EXIT
					;
	LD	A,(CMD)			; WHAT COMMAND IS PENDING?
	OR	A			; SET FLAGS
	JP	DOSO4			; NO, MUST BE READ OR WRITE COMMAND
DSKEXIT:	
	LD	HL,FLATCH_STORE		; POINT TO FLATCH
	SET	1,(HL)			; SET MOTOR OFF
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
					;
	OR	0FFH			; SET -1 IF ERROR
	RET

RESULT:
	LD	C,07H			; LOAD C WITH NUMBER OF STATUS BYTES
	LD	HL,ST0			; POINT TO STATS STORAGE
RS3:
	CALL	GFDATA			; GET FIRST BYTE
	LD	(HL),A			; SAVE IT
	INC	HL			; POINTER++
	DEC	C			; CC-1
	JP	NZ,RS3			; LOOP TIL C0
	LD	A,(ST0)			; LOAD STS0
	AND	0F8H			; MASK OFF DRIVE #
	LD	B,A			; PARK IT
	LD	A,(ST1)			; LOAD STS1
	OR	B			; ACC OR B ->ACC IF 0 THEN SUCCESS
					;
RSTEXIT:
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	LD	HL,FLATCH_STORE		; POINT TO FLATCH
	SET	1,(HL)			; SET MOTOR OFF
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
					;
	RET				; DONE RETURN TO CALLER 
	
	
DOSO4:
	LD	HL,(TEMPW)		; GET BUFFER ADDRESS TO HL
	
	LD	A,(SECSIZ)		; XFERLEN
	LD	C,A			; C WILL BE THE NUMBER OF TRANSACTIONS
					; DIVIDED BY 4
					;
	LD	A,(CMD)			;
	CALL	PFDATA			; PUSH COMMAND TO I8272
	LD	A,(UNIT)		;
	CALL	PFDATA			; 
	LD	A,(PTRACK)		;
	CALL	PFDATA			; 
	LD	A,(HEAD)		;
	CALL	PFDATA			; 
	LD	A,(SECTOR)		;
	CALL	PFDATA			; 
	LD	A,(DENS)		;
	CALL	PFDATA			; WHAT DENSITY
	LD	A,(EOTSEC)		;
	CALL	PFDATA			; ASSUME SC (SECTOR COUNT)  EOT
	LD	A,(GAP)			;
	CALL	PFDATA			; WHAT GAP IS NEEDED
	LD	A,(DTL)			; DTL, IS THE LAST COMMAND BYTE TO I8272
	CALL	PFDATAS
;

	;
; PERFORM READ
; LOOP EXECUTES 4X, THIS ALLOWS C RATHER THAN BC AS COUNTER
; SAVING A FEW TSTATES  MAKES UP TO 1024 BYTE SECTORS POSSIBLE.
; FROM READ TO READ MUST NOT EXCEED 25US WORST CASE MIN 
; (76T STATES FOR 3MHZ 8085) OR (100 T STATES FOR 4MHZ Z80)
;

RDD_POLL:
FDC_READP0:
	IN	A,(FMSR)		;
	OR	A			; TEST IF BYTE READY RQM1
	JP	P,FDC_READP0		;	
					;
	AND	020h			;
	JP	Z,DSKOPEND		;JUMP IF IN RESULTS MODE
					;
	IN	A,(FDATA)		;
	LD	(HL),A			;
	INC	HL			;

FDC_READP1:
	IN	A,(FMSR)		;
	OR	A			;
	JP	P,FDC_READP1		;
					;
	AND	020h			;
	JP	Z,DSKOPEND		;
					;
	IN	A,(FDATA)		;
	LD	(HL),A			;
	INC	HL			;
					;
FDC_READP2:
	IN	A,(FMSR)		;
	OR	A			;
	JP	P,FDC_READP2		;
					;
	AND	020h			;
	JP	Z,DSKOPEND		;
					;
	IN	A,(FDATA)		;
	LD	(HL),A			;
	INC	HL			;
					;
FDC_READP3:
	IN	A,(FMSR)		; 11
	OR	A			; 4
	JP	P,FDC_READP3		; 10
					;
	AND	020h			; 7
	JP	Z,DSKOPEND		; 10
					;
	IN	A,(FDATA)		; 11
	LD	(HL),A			; 10
	INC	HL			; 11
					;
	DEC	C			; 4
	JP	NZ,FDC_READP0		; 11
DSKOPEND:
	LD	HL,FLATCH_STORE		; POINT TO FLATCH
	SET	0,(HL)			; SET TC
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
	NOP				;
	NOP				; 2 MICROSECOND DELAY
	RES	0,(HL)			; RESET TC
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
	NOP				;
	NOP				; 2 MICROSECOND DELAY
	NOP				;
	NOP				; 2 MICROSECOND DELAY
	SET	1,(HL)			; TURN OFF MOTOR
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
	JP	RESULT			; GET STATUS BYTES <RESULT PHASE>
	
		
;__SETTRACK__________________________________________________________________________________________________________________________ 
;
; 	SEEK TO A TRACK ON GIVEN UNIT
; 	A: TRACK #
;________________________________________________________________________________________________________________________________
;
SETTRACK:
	LD	A,(FTRACK)		; GET CURRENT HEAD TRACK
	LD	C,A
	LD	A,(PTRACK)		; GET TRACK
	OR	A			; SET FLAGS
	JP	Z,RECAL			; IF 0 PERFORM RECAL INSTEAD OF SEEK
	CP	C			;
	JP	Z,WAINT			; ALREADY THERE, ABORT
	LD	(FTRACK),A		; STORE TRACK
	LD	A,0FH			; SEEK COMMAND
	CALL	PFDATA			; PUSH COMMAND
	LD	A,(UNIT)		; SAY WHICH UNIT
	CALL	PFDATA			; SEND THAT
	LD	A,(PTRACK)		; TO WHAT TRACK
	CALL	PFDATA			; SEND THAT TOO
	JP	WAINT			; WAIT FOR INTERRUPT SAYING DONE
RECAL:
	LD	A,00H			;
	LD	(FTRACK),A		; STORE TRACK
	LD	A,07H			; RECAL TO TRACK 0
	CALL	PFDATA			; SEND IT
	LD	A,(UNIT)		; WHICH UNIT
	CALL	PFDATA			; SEND THAT TOO
;
WAINT:
;
	CALL	DELAYHSEC		; DELAY TO LET HEADS SETTLE BEFORE READ
					;
					; WAIT HERE FOR INTERRPT SAYING DONE
					; LOOP TIL INTERRUPT
	CALL	CHECKINT		; CHECK INTERRUPT STATUS
;
	RET
	
	
;__CYCLEFLOPPY__________________________________________________________________________________________________________________________ 
;
; 	SEEK TO TRACK 0, THEN BACK TO THE SELECTED TRACK    
;	THIS CAN BE USED ON AN ERROR CONDITION TO VERIFY THAT HEAD IS ON SELECTED TRACK
; 	
;________________________________________________________________________________________________________________________________
;
CYCLEFLOPPY:
	PUSH	AF			; STORE	AF
	PUSH	HL			; STORE	HL
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	CALL	CHECKINT		; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
	LD	HL,FLATCH_STORE		; POINT TO FLATCH
	RES	1,(HL)			; SET MOTOR ON
	CALL	OUTFLATCH		; OUTPUT TO CONTROLLER
	NOP				;
	NOP				;
	CALL	RECAL			;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	CALL	RECAL			;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	CALL	SETTRACK		;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	CALL	DELAYHSEC		;
	POP	HL			;
	POP	AF			; RESTORE AF
	RET	

;__PFDATAS__________________________________________________________________________________________________________________________ 
;
; WRITE A COMMAND OR PARAMETER SEQUENCE
;
; TRANSFERS ARE SYNCHONIZED BYT MSR D7 <RQM> AND D6 <DIO>
;	RQM  DIO
;	0	0	BUSY
;	1	0	WRITE TO DATA REGISTER PERMITTED
;	1	1	BYTE FOR READ BY HOST PENDING
;	0	1	BUSY
;
;________________________________________________________________________________________________________________________________
;
PFDATAS:
	PUSH	AF			; STORE AF
PFDS1:
	IN	A,(FMSR)		; READING OR WRITING IS KEYS TO D7 RQM
	AND	80H			; MASK OFF RQM BIT 
	JP	Z,PFDS1			; WAIT FOR RQM TO BE TRUE 
	IN	A,(FMSR)		; READ STATUS
	AND	40H			; WAITING FOR INPUT?
	CALL	NZ,ERRORT		; NO, SIGNAL ERROR
	POP	AF			; RESTORE AF
	OUT	(FDATA),A		; OUTPUT A TO CONTROLLER
	RET		
	
;__PFDATA__________________________________________________________________________________________________________________________ 
;
; WRITE A COMMAND OR PARAMETER SEQUENCE
;
; TRANSFERS ARE SYNCHONIZED BYT MSR D7 <RQM> AND D6 <DIO>
;	RQM  DIO
;	0	0	BUSY
;	1	0	WRITE TO DATA REGISTER PERMITTED
;	1	1	BYTE FOR READ BY HOST PENDING
;	0	1	BUSY
;
;________________________________________________________________________________________________________________________________
;
PFDATA:
	PUSH	AF			; STORE AF
PFD1:
	IN	A,(FMSR)		; READING OR WRITING IS KEYS TO D7 RQM
	AND	80H			; MASK OFF RQM BIT 
	JP	Z,PFD1			; WAIT FOR RQM TO BE TRUE 
	IN	A,(FMSR)		; READ STATUS
	AND	40H			; WAITING FOR INPUT?
	CALL	NZ,ERRORT		; NO, SIGNAL ERROR
	POP	AF			; RESTORE AF
	OUT	(FDATA),A		; OUTPUT A TO CONTROLLER
	JP	DELAY24			; WAIT 24 US BEFORE READING FMSR AGAIN


;__CHECKINT__________________________________________________________________________________________________________________________ 
;
; CHECK FOR ACTIVE FDC INTERRUPTS BEFORE GIVING I8272 COMMANDS
; POLL RQM FOR WHEN NOT BUSY AND THEN SEND FDC
; SENSE INTERRUPT COMMAND   IF IT RETURNS WITH NON ZERO
; ERROR CODE, PASS BACK TO CALLING ROUTINE FOR HANDLING
;________________________________________________________________________________________________________________________________
;
CHECKINT:
	IN	A,(FMSR)		; READING OR WRITING IS KEYS TO D7 RQM
	AND	80H			; MASK OFF RQM BIT
	JP	Z,CHECKINT		; WAIT FOR RQM TO BE TRUE  WAIT UNTIL DONE
	IN	A,(FMSR)		; READ STATUS
	AND	40H			; WAITING FOR INPUT?
	JP	NZ,CHECKINTDONE		; NO, SIGNAL ERROR
	CALL	SENDINT			; SENSE INTERRUPT COMMAND
CHECKINTDONE:
	RET				;
	

;__DELAYHSEC__________________________________________________________________________________________________________________________ 
;
; DELAY FOR 1/2 SECOND
;________________________________________________________________________________________________________________________________
;		
DELAYHSEC:
	LD	HL,00000H		; 65536
DELDM:
	NOP				; (4 T) 
	NOP				; (4 T)
	NOP				; (4 T)
	NOP				; (4 T)
	DEC	L			; (6 T)
	JP	NZ,DELDM		; (10 T) 24 T  8 MICROSECONDS AT 4 MHZ
	DEC	H			; (6 T)
	JP	NZ,DELDM		; (10 T) (8 US * 256) * 256  524288 US   5 SECONDS
	RET

;__ERRORT__________________________________________________________________________________________________________________________ 
;
; ERROR HANDLING
;________________________________________________________________________________________________________________________________
;			
ERRORT:
	IN	A,(FDATA)		; CLEAR THE JUNK OUT OF DATA REGISTER
	IN	A,(FMSR)		; CHECK WITH RQM
	AND	80H			; IF STILL NOT READY, READ OUT MORE JUNK
	JP	Z,ERRORT		;
	LD	A,0FFh			; RETURN ERROR CODE -1
					;
	RET
;__SENDINT__________________________________________________________________________________________________________________________ 
;
; SENSE INTERRUPT COMMAND
;________________________________________________________________________________________________________________________________
;			
SENDINT:
	LD	A,08H			; SENSE INTERRUPT COMMAND
	CALL	PFDATA			; SEND IT
	CALL	GFDATA			; GET RESULTS
	LD	(ST0A),A		; STORE THAT
	AND	0C0h			; MASK OFF INTERRUPT STATUS BITS
	CP	080h			; CHECK IF INVALID COMMAND
	JP	Z,ENDSENDINT		; YES, EXIT
	CALL	GFDATA			; GET ANOTHER (STATUS CODE 1)
	LD	(ST1A),A		; SAVE THAT
	LD	A,(ST0A)		; GET FIRST ONE
	AND	0C0H			; MASK OFF ALL BUT INTERRUPT CODE 00 IS NORMAL
ENDSENDINT:
	RET				;ANYTHING ELSE IS AN ERROR


;__GFDATA__________________________________________________________________________________________________________________________ 
;
; GET DATA FROM FLOPPY CONTROLLER
;
; TRANSFERS ARE SYNCHONIZED BYT MSR D7 <RQM> AND D6 <DIO>
;	RQM  DIO
;	0	0	BUSY
;	1	0	WRITE TO DATA REGISTER PERMITTED
;	1	1	BYTE FOR READ BY HOST PENDING
;	0	1	BUSY
;
;________________________________________________________________________________________________________________________________
;		
GFDATA:
	IN	A,(FMSR)		; READ STATUS BYTE
	AND	80H			; MASK OFF RQM
	JP	Z,GFDATA		; LOOP WHILE BUSY
	IN	A,(FMSR)		; READ STSTUS BUTE
	AND	40H			; MASK OFF DIO
	CALL	Z,ERRORT		; IF WRITE EXPECTED RUN ERRORRT
	IN	A,(FDATA)		; READ DATA
	JP	DELAY24			; WAIT 24 US BEFORE READING FMSR AGAIN

	 .IF DSKY
		
;__DSPSECTOR_______________________________________________________________________________________________________________________ 
;
;	DISPLAY SECTOR IN HL ON FRONT PANEL
;________________________________________________________________________________________________________________________________
;
DSPSECTOR:
	PUSH	BC			; STORE BC
	PUSH	HL			; STORE HL
	LD	A,H			; DISPLAY HIGH BYTE, HIGH NIBBLE
	SRL 	A			;	
	SRL 	A			;	
	SRL 	A			;	
	SRL 	A			;	
	AND	0FH			;
	CALL 	DECODEDISPLAY		;
	LD	(SEC+3),A		;
	LD      A,H			; DISPLAY HIGH BYTE, LOW NIBBLE
	AND	0FH			;
	CALL 	DECODEDISPLAY		;
	LD	(SEC+2),A		;
	LD	A,L			; DISPLAY LOW BYTE, HIGH NIBBLE
	AND	0F0H			;
	SRL 	A			;	
	SRL 	A			;	
	SRL 	A			;	
	SRL 	A			;		
	AND	0FH			;
	CALL 	DECODEDISPLAY		;
	LD	(SEC+1),A		; DISPLAY LOW BYTE, LOW NIBBLE
	LD      A,L			;
	AND	0FH			;
	CALL 	DECODEDISPLAY		;
	LD	(SEC),A			;
	LD	HL,SEC			; DISPLAY PROMPT
	CALL	SEGDISPLAY		; 
	POP	HL			; RESTORE HL
	POP	BC			; RESTORE BC
	RET
	 .ENDIF

	
;___IDE_READ_SECTOR______________________________________________________________________________________
;
;	READ IDE SECTOR
;________________________________________________________________________________________________________			
IDE_READ_SECTOR:
	CALL	IDE_WAIT_BUSY_READY 	; MAKE SURE DRIVE IS READY TO PROCEED
	RET	NC			; ERROR, RETURN
	CALL	IDE_SETUP_LBA		; TELL DRIVE WHAT SECTOR IS REQUIRED
	LD	A,20H			;
	OUT	(IDESTTS),A		; 020h = IDE 'READ SECTOR' COMMAND 
IDE_SREX:				;
	CALL	IDE_WAIT_BUSY_READY	; MAKE SURE DRIVE IS READY TO PROCEED
	RET	NC			; ERROR, RETURN
	CALL	IDE_TEST_ERROR		; ENSURE NO ERROR WAS REPORTED
	RET	NC			; ERROR, RETURN
	CALL	IDE_WAIT_BUFFER		; WAIT FOR FULL BUFFER SIGNAL FROM DRIVE
	RET	NC			; ERROR, RETURN
	CALL	IDE_READ_BUFFER		; GRAB THE 256 WORDS FROM THE BUFFER
	SCF				; CARRY = 1 ON RETURN = OPERATION OK
IDE_READ_SECTOR_OK:			;
	SCF				; CARRY = 1 ON RETURN = OPERATION OK
	RET
;___IDE_READ_BUFFER_______________________________________________________________________________________
;
;	READ IDE BUFFER
;________________________________________________________________________________________________________			
IDE_READ_BUFFER:
	LD	B,0			; 256 WORDS (512 BYTES PER SECTOR)
IDEBUFRD:				;
	IN	A,(IDELO)		; LOW BYTE OF WORD FIRST	
	LD	(IX+0),A		;
	INC	IX			;
	IN	A,(IDEHI)		; THEN HIGH BYTE OF WORD
	LD	(IX+0),A		;
	INC	IX			;
	DEC	B			;
	JP	NZ,IDEBUFRD		;
	RET
	
;___IDE_SETUP_LDA________________________________________________________________________________________
;
;	SETUP IDE DRIVE FOR LDA OPERATION
;________________________________________________________________________________________________________			
IDE_SETUP_LBA:
	LD	A,L			; LOAD LBA REGISTER 0 WITH SECTOR ADDRESS TO READ
	LD	(IDE_LBA0),A		;
	LD	A,H			; LOAD LBA REGISTER 1 WITH SECTOR ADDRESS TO READ
	LD	(IDE_LBA1),A		;
	LD	A,0000H		 	; LOAD LBA REGISTER 2 WITH SECTOR ADDRESS TO READ
	LD	(IDE_LBA2),A		;
	AND	000001111b		; ONLY LOWER FOUR BITS ARE VALID
	ADD	A,011100000b		; ENABLE LBA BITS 5:7=111 IN IDE_LBA3
	LD	(IDE_LBA3),A		;
					; READ IDE HD SECTOR
	LD	A,1			;
	OUT	(IDESECTC),A		; SET SECTOR COUNT = 1	
					;	
	LD	A,(IDE_LBA0)		;
	OUT	(IDESECTN),A		; SET LBA 0:7
					;
	LD	A,(IDE_LBA1)		;
	OUT	(IDECYLLO),A		; SET LBA 8:15
					;
	LD	A,(IDE_LBA2)		;
	OUT	(IDECYLHI),A		; SET LBA 16:23
					;
	LD	A,(IDEDEVICE)		; SELECTS DEVICE
	LD	C,A			;				
	LD	A,(IDE_LBA3)		;
	AND	000001111b		; LOWEST 4 BITS USED ONLY
	OR	011100000b		; TO ENABLE LBA MODE
	OR	C			;
	OUT	(IDEHEAD),A		; SET LBA 24:27 + BITS 5:7=111
	RET	
	
;___IDE_TEST_ERROR_______________________________________________________________________________________
;
;	CHECK FOR IDE ERROR CONDITION
;________________________________________________________________________________________________________			
IDE_TEST_ERROR:
	SCF				;
	IN	A,(IDESTTS)		;
	LD	B,A			; 
	AND	000000001b		; TEST ERROR BIT
	SCF				; 
	RET	Z			;
	LD	A,B			; 
	AND	000100000b		;
	SCF				;
	JP	NZ,IDE_ERR		; TEST WRITE ERROR BIT
	IN	A,(IDEERR)		; READ ERROR FLAGS
IDE_ERR:
	OR	A			; CARRY 0 = ERROR
	RET				; IF A = 0, IDE BUSY TIMED OUT

;___IDE_WAIT_BUFFER_______________________________________________________________________________________
;
;	WAIT FOR DATA BUFFER READY
;________________________________________________________________________________________________________			
IDE_WAIT_BUFFER:
	LD	DE,0			;
IDE_WDRQ:				;
	LD	B,5			;
IDE_BLP:				;
	DEC	B			;
	JP	NZ,IDE_BLP		;	
	INC	DE			;
	LD	A,D			;
	OR	E			;
	JP	Z,IDE_TO2		;
	IN	A,(IDESTTS)		; WAIT FOR DRIVE'S 512 BYTE READ BUFFER 
	AND	000001000b		; TO FILL (OR READY TO FILL)
	JP	Z,IDE_WDRQ		;
	SCF				; CARRY 1 = OK
	RET				;
IDE_TO2:				;
	XOR	A			; CARRY 0 = TIMED OUT
	RET				;
	
	
	
	
	
;
;__INIT_UART_____________________________________________________________________________________________________________________ 
;
;	INITIALIZE UART
;	PARAMS:	SER_BAUD NEEDS TO BE SET TO BAUD RATE
;	1200:	96	 = 1,843,200 / ( 16 X 1200 )
;	2400:	48	 = 1,843,200 / ( 16 X 2400 )
;	4800:	24	 = 1,843,200 / ( 16 X 4800 )
;	9600:	12	 = 1,843,200 / ( 16 X 9600 )
;	19K2:	06	 = 1,843,200 / ( 16 X 19,200 )
;	38K4:	03	
;	57K6:	02
;	115K2:	01	
;
;_________________________________________________________________________________________________________________________________
;
INIT_UART:
	LD	A,80H			;
	OUT	(UART3),A		; SET DLAB FLAG
	LD	A,(SER_BAUD)		;
	OUT	(UART0),A		;
	LD	A,00H			;
	OUT	(UART1),A		;
	LD	A,03H			;
	OUT	(UART3),A		; SET 8 BIT DATA, 1 STOPBIT
	RET
	
	
	 .IF UART	
;__SPACE_________________________________________________________________________________________________________________________ 
;
;	PRINT A SPACE CHARACTER ON THE SERIAL PORT
;________________________________________________________________________________________________________________________________
;
SPACE:
	PUSH	AF			; STORE AF
	LD	A,20H			; LOAD A "SPACE"
	CALL	COUT			; SCREEN IT
	POP	AF			; RESTORE AF
	RET				; DONE


;__COUT__________________________________________________________________________________________________________________________ 
;
;	WRITE THE VALUE IN "A" TO THE SERIAL PORT
;________________________________________________________________________________________________________________________________
;
COUT:
	PUSH   AF			; STORE AF
TX_BUSYLP:
	IN	A,(UART5)		; READ LINE STATUS REGISTER
	BIT	5,A			; TEST IF UART IS READY TO SEND
	JP	Z,TX_BUSYLP		; IF NOT REPEAT
	POP	AF			; RESTORE AF
	OUT	(UART0),A		; THEN WRITE THE CHAR TO UART
	RET				; DONE


;__CRLF__________________________________________________________________________________________________________________________ 
;
;	SEND CR & LF TO THE SERIAL PORT
;________________________________________________________________________________________________________________________________
;
CRLF:
	PUSH	HL			; PROTECT HL FROM OVERWRITE
	LD	HL,TCRLF		; LOAD MESSAGE POINTER
	CALL	MSG			; SEBD MESSAGE TO SERIAL PORT
	POP	HL			; PROTECT HL FROM OVERWRITE
	RET				;
;__MSG___________________________________________________________________________________________________________________________ 
;
;	PRINT A STRING  TO THE SERIAL PORT
;________________________________________________________________________________________________________________________________
;
MSG:

TX_SERLP:
	LD	A,(HL)			; GET CHARACTER TO A
	CP	END			; TEST FOR END BYTE
	JP	Z,TX_END		; JUMP IF END BYTE IS FOUND
	CALL	COUT			;
	INC	HL			; INC POINTER, TO NEXT CHAR
	JP	TX_SERLP		; TRANSMIT LOOP
TX_END:
	RET				;ELSE DONE


TCRLF:
	 .db  	CR,LF,END

;__CRLFA_________________________________________________________________________________________________________________________ 
;
;	PRINT COMMAND PROMPT TO THE SERIAL PORT
;________________________________________________________________________________________________________________________________
;
CRLFA:
	PUSH	HL			; PROTECT HL FROM OVERWRITE
	LD	HL,PROMPT		;
	CALL	MSG			;
	POP	HL			; PROTECT HL FROM OVERWRITE
	RET				; DONE
;__GETLN_________________________________________________________________________________________________________________________ 
;
;	READ A LINE(80) OF TEXT FROM THE SERIAL PORT, HANDLE <BS>, TERM ON <CR> 
;       EXIT IF TOO MANY CHARS    STORE RESULT IN HL.  CHAR COUNT IN C.
;________________________________________________________________________________________________________________________________
;
GETLN:
	LD	C,00H			; ZERO CHAR COUNTER
	PUSH	DE			; STORE DE
GETLNLOP:
	CALL	KIN			; GET A KEY
	CALL	COUT			; OUTPUT KEY TO SCREEN
	CP	CR			; IS <CR>?
	JR	Z,GETLNDONE		; YES, EXIT 
	CP	BS			; IS <BS>?
	JR	NZ,GETLNSTORE		; NO, STORE CHAR
	LD	A,C			; A=C
	CP	0			;
	JR	Z,GETLNLOP		; NOTHING TO BACKSPACE, IGNORE & GET NEXT KEY
	DEC	HL			; PERFORM BACKSPACE
	DEC	C			; LOWER CHAR COUNTER	
	LD	A,0			;
	LD	(HL),A			; STORE NULL IN BUFFER
	LD	A,20H			; BLANK OUT CHAR ON TERM
	CALL	COUT			;
	LD	A,BS			;
	CALL	COUT			;
	JR	GETLNLOP		; GET NEXT KEY
GETLNSTORE:
	LD	(HL),A			; STORE CHAR IN BUFFER
	INC	HL			; INC POINTER
	INC	C			; INC CHAR COUNTER	
	LD	A,C			; A=C
	CP	4DH			; OUT OF BUFFER SPACE?
	JR	NZ,GETLNLOP		; NOPE, GET NEXT CHAR
GETLNDONE:
	LD	(HL),00H		; STORE NULL IN BUFFER
	POP	DE			; RESTORE DE
	RET				;
;__KIN___________________________________________________________________________________________________________________________ 
;
;	READ FROM THE SERIAL PORT AND ECHO & CONVERT INPUT TO UCASE
;________________________________________________________________________________________________________________________________
;
KIN:
	IN	A,(UART5)		; READ LINE STATUS REGISTER
	BIT	0,A			; TEST IF DATA IN RECEIVE BUFFER
	JP	Z,KIN			; LOOP UNTIL DATA IS READY
	IN	A,(UART0)		; THEN READ THE CHAR FROM THE UART
	AND	7FH			; STRIP HI BIT
	CP	'A'			; KEEP NUMBERS, CONTROLS
	RET	C			; AND UPPER CASE
	CP	7BH			; SEE IF NOT LOWER CASE
	RET	NC			; 
	AND	5FH			; MAKE UPPER CASE
	RET


	 .ENDIF


;
;__TEXT_STRINGS_________________________________________________________________________________________________________________ 
;
;	SYSTEM TEXT STRINGS
;_____________________________________________________________________________________________________________________________
;
	 .IF DSKY
BOOT:
	 .DB 	00H,00H,80H,80H,094H,09DH,09DH,09FH
	 .ENDIF
	 .IF UART
PROMPT:
	 .db  	CR,LF,'>',END

MSG_START:
	 .DB	LF,CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"N8VEM BOOT LOADER"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN

	 .db  	"0->DSKY MONITOR"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"1->SERIAL MONITOR"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"2->BOOT CP/M ROM IMAGE"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"3->BOOT FLOPPY UNIT 0"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"4->BOOT FLOPPY UNIT 1"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"5->BOOT IDE PRIMARY"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"6->BOOT IDE SECONDARY"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"7->BOOT ATAPI PRIMARY"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .db  	"8->BOOT ATAPI SECONDARY"
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .DB	LF, CR			; LINE FEED AND CARRIAGE RETURN
	 .DB	END			; LINE TERMINATOR
	 .ENDIF

;_ATAPI COMMAND PACKETS______________________________________________________________________________________________________
; 
;
ATAPI_REQUEST_SENSE
	 .DB	03H,00H,00H,00H,011H,00H,00H,00H,00H,00H,00H,00H
	
	
	 .IF DSKY	
;_KB DECODE TABLE__________________________________________________________________________________________________________
; 
;
KB_DECODE:
;                0  1   2   3   4   5   6   7   8   9   A   B   C   D   E   F
	 .DB	41H,02H,42H,82H,04H,44H,84H,08H,48H,88H,10H,50H,90H,20H,60H,0A0H
;               FW  BK  CL  EN  DP  EX  GO  BO
	 .DB	01H,81H,0C1H,0C2H,0C4H,0C8H,0D0H,0E0H
	
	
;
; F-KEYS,
; FW = FORWARD
; BK = BACKWARD
; CL = CLEAR
; EN = ENTER
; DP = DEPOSIT (INTO MEM)
; EX = EXAMINE (MEM)
; GO = GO
; BO = BOOT
;_________________________________________________________________________________________________________________________
;_HEX 7_SEG_DECODE_TABLE__________________________________________________________________________________________________
; 
; 0,1,2,3,4,5,6,7,8,9,A,B,C,D,E,F, ,-
; AND WITH 7FH TO TURN ON DP 
;_________________________________________________________________________________________________________________________
SEGDECODE:
	 .DB	0FBH,0B0H,0EDH,0F5H,0B6H,0D7H,0DFH,0F0H,0FFH,0F7H,0FEH,09FH,0CBH,0BDH,0CFH,0CEH,080H,084H,00H,0EEH,09DH
	
	 .ENDIF
			
	 .ORG	009FFh
FLAG	 .DB	0FFh


;
;__WORK_AREA___________________________________________________________________________________________________________________ 
;
;	RESERVED RAM FOR MONITOR WORKING AREA
;_____________________________________________________________________________________________________________________________
;

KEYBUF:		.EQU	0A000H
SER_BAUD:	.EQU	0A050H
DISPLAYBUF:	.EQU	0A051H
IDEDEVICE:	.EQU	0A059H
CMD:		.EQU	0A05AH
UNIT:		.EQU	0A05BH
HEAD:		.EQU	0A05CH
DENS:		.EQU	0A05DH
EOTSEC:		.EQU	0A05EH
GAP:		.EQU	0A05FH
SECSIZ:		.EQU	0A060H
DTL:		.EQU	0A061H
SRTHUT:		.EQU	0A062H
HLT:		.EQU	0A063H
MIN:		.EQU	0A064H
PRE:		.EQU	0A065H
ST0:		.EQU	0A066H
ST1:		.EQU	0A067H
ST2:		.EQU	0A068H
SCYL:		.EQU	0A069H
SHEAD:		.EQU	0A06AH
SREC:		.EQU	0A06BH
SNBIT:		.EQU	0A06CH
ST0A:		.EQU	0A06DH
ST1A:		.EQU	0A06EH
RETRY		.EQU	0A06FH
RETRY1		.EQU	0A070H
FLATCH_STORE:	.EQU	0A071H
SECTOR		.EQU	0A072H
PTRACK:		.EQU	0A073H
FTRACK:		.EQU	0A075H
TEMPW:		.EQU	0A077H
IDE_LBA0:	.EQU	0A079H
IDE_LBA1:	.EQU	0A07AH
IDE_LBA2:	.EQU	0A07BH
IDE_LBA3:	.EQU	0A07CH
SECTOR_BUFFER:	.EQU	0A07DH
READ_DISK_PACKET:	.EQU	0A280H
SEC:		.EQU	0A290H
 



 .END
 
 
 

 