; XXX romdisk
;
;  scan rom banks 1 ... last
;  stop when 1st byte not 0 nor FF
;  set rom bank0 = boot
;  zeroed physical bank0 can not boot... hmmm.

	include "memory.inc"
	include "io.inc"

;----------------------------------------------------------------------
	org RAM_BIOS

boote:
	jp boot       ; at end, will be used for buffers
wboote:
	jp wboot
	jp const
	jp conin
	jp conout
	jp list
	jp punch
	jp reader
	jp home
	jp seldsk
	jp settrk
	jp setsec
	jp setbuf     ; dma ? what dma ?
	jp read
	jp write
	jp listst     ; liszt
	jp sectran

	call oops     ; leave traces
	call oops     ;  if unexpectedly used
	call oops

;----------------------------------------------------------------------
; 00      boot
; 01      romdisk start
; ..
; 80      RAM enable
; ..
; FC      ramdisk start
; FD
; FE      TPAlower
; FF      TPAupper+system
;
; XXX in SELMEM/SETBNK (cpm3?) counting is different,
; XXX SELMEM 1 would mean TPA.

oops:
	di
	ex af, af'

	ld a, 0           ; allow patching
	out (BANK), a     ; rom bank 0. cbank not touched

	ex af, af'
	jp ROM_MONITOR

reload_bios:
	di
	ld a, 0
	out (BANK), a
	jp 0

bank_std:
	ld a, BANK_TPA

bank_set:           ; XXX if macros were used, loss of stack would not hurt
	ld (cbank), a
	out (BANK), a
	ret

peek:                ; a = (a:de)
	call iff_disable
	call bank_set

	ld a, (de)
	push af
	call bank_std
	pop af
	jp iff_restore

;----------------------------------------------------------------------

iff_disable:
	di
	push hl
	ld hl, iff
	inc (hl)
	pop hl
	ret

iff_restore:
	push hl
	ld hl, iff
	dec (hl)
	pop hl
	ret nz
	ei
	ret

iff_enable:
	push af
	xor a
	ld (iff), a
	pop af
	ei
	ret

;----------------------------------------------------------------------

wboot:
	ld sp, 0              ; there is space above bss
	di
	im 1
	call bank_std

	ld hl, 0              ; restarts. hl += 8 in patch_jp
	ld bc, wboote
	call patch_jp         ; RST 0 JP wboote

	ld bc, oops           ; XXX drop to monitor
	call patch_jp         ; RST 1 JP oops
	call patch_jp         ; RST 2 JP oops
	call patch_jp         ; RST 3 JP oops
	call patch_jp         ; RST 4 JP oops
	call patch_jp         ; RST 5 JP oops
	call patch_jp         ; RST 6 JP oops

; XXX	ld bc, interrupt
	call patch_jp         ; RST 7 JP interrupt

	call iff_enable       ; XXX UART_IER

	; load ccp and bdos XXX check sd if replacements there

	ld a, 0               ; shuffled rom banks XXX ?
	call iff_disable
	call bank_set

	ld hl, ROM_CPM
	ld de, RAM_CCP
	ld bc, RAM_BIOS - RAM_CCP
	ldir
	call bank_std
	call iff_enable

cold_once:
	ld a, 'C'             ; A:CSTARTUP or A:WSTARTUP
	push af               ; pass it in a register too
	ld (RAM_CCP+10), a    ; patch into CCP input buffer
	ld a, 'W'
	ld (cold_once+1), a   ; XXX not rommable

	ld hl, BDOS
	ld bc, RAM_BDOS
	call patch_jp         ; 0005 JP bdos

	ld bc, 0080h          ; default read/write buffer
	call setbuf

	ld a, (CDISK)         ; does DISK exist ?
	and 0Fh
	call seldsk_a
	ld a, h
	or l
	ld a, (CDISK)
	jr nz, L0             ; DISK ok
	and 0F0h
	ld (CDISK), a         ; reset DISK, keep USER
L0:
	pop bc                ; b is 'C' or 'W'
	ld c, a
	jp RAM_CCP            ; gone

;----------------------------------------------------------------------
; list    11______ UL1: NUL:
;         10      *LPT: to PPI_B
;         01       CRT:
;         00       TTY:
; punch   __11____*UP2: same as LPT: but ignores busy
; auxout    10     UP1:
;           01     PTP:
;           00     TTY:
; reader  ____11__*UR2: yymmddHHMMSS. idea copied from Alspaugh bios
; auxin       10   UR1: from PPI_A
;             01   PTR:
;             00   TTY:
; console ______11 UC1:
;               10 BAT: --- in from READER, out to LIST XXX
;               01 CRT:
;               00*TTY: to/from UART
;
;  and 0C0h
;  jr z, ttyout
;  jp pe, ul1
;  jp p, crtout
; lpt:
;
;  and 0Ch
;  jr z, ttyin
;  jp pe, ur2
;  and 08h
;  jr nz, ur1
; ptr:

;----------------------------------------------------------------------
; TTY: ~ CON:

conin:
ttyin:
	call ttyst
	jr z, ttyin           ; wait a character
	in a, (UART_DAT)
	ret                   ; XXX n tuskin kelpaa

conout:
ttyout:
	call ttyost
	jr z, ttyout          ; wait it
	ld a, c
	out (UART_DAT), a
	ret

ttyost:
	in a, (UART_LSR)      ; XXX acknowledges line status interrupt
	and 20h               ; THRE
	ret z
	ld a, 0FFh
	ret

const:
ttyst:
	in a, (UART_LSR)
	and 01h               ; receive data available
	ret z
	ld a, 0FFh
	ret 

;----------------------------------------------------------------------
; LPT: ~ LST:

listst:
	ld a, (IOBYTE)      ; TTY: CRT: LPT: UL1:
	add a, a            ; set carry and sign
	jr nc, ttyost       ; CRT: or TTY:
	jp m, rdy           ; NUL: (nz also set, when minus)
lptst:
	in a, (PPI_C)       ; BUSY PPI_C.7
	cpl                 ; positive busy, make it "positive ready"
	and 80h             ; ready?
	ret z
rdy:
	ld a, 0FFh
	ret

list:
	ld a, (IOBYTE)
	add a, a
	jr nc, ttyout       ; CRT: or TTY:
	ret m               ; NUL:
lpt:
	call lptst
	jr z, lpt           ; until ready
punch:                  ; same as lpt but ignores busy
	ld a, c
	out (PPI_B), a      ; setup min 500ns (from random Centronics docs)
	ld a, 0             ; clr-bit PPI_C.0 nSTROBE
	out (PPI_CTRL), a   ; strobe width min 500ns
	inc a               ; set-bit, same bit XXX or maybe xor 1
	out (PPI_CTRL), a   ; hold min 500ns
	ret                 ; BUSY goes up in 500ns

reader:
	ld a, (IOBYTE)
	and 4
	jr nz, ur2
	in a, (PPI_C)       ; UR1: XXX handshaking ?
	ld c, a
	in a, (PPI_A)
	ret

ppi_init:
	ld a, 98h           ; mode 0 (1 mm a u m b l) A+Cupper in, B+Clower out
	out (PPI_CTRL), a   ; output pins zeroed by mode set
	ld a, 1             ; set-bit PPI_C.0 (0000 bit set/#clr)
	out (PPI_CTRL), a   ; nSTROBE inactive XXX might print binary 0x00
	ret

;----------------------------------------------------------------------
; UR2:
; yymmddHHMMSS and ^Z from RTC

ur2:
	ld hl, rdr_buf + 6  ; rdr_buf: sec min hour day mon year count
	dec (hl)
	ld a, 1Ah           ; ^Z
	ret z               ;  between each timestamp
	jp p, rdr_1         ; something still buffered

	ld hl, rdr_buf      ; fresh values to rdr_buf, S M H d m weekday y
	call clock_get      ; hl += 7

	dec hl              ; squeeze out weekday
	ld a, (hl)
	ld (hl), 12         ; 7th byte is count instead
	dec hl
	ld (hl), a          ; year in 6th byte
rdr_1:
	ld hl, rdr_buf
	ld b, 6
	ld a, '0'           ; 30h, rld keeps the 3
rdr_2:
	rld                 ; push nybbles
	inc hl
	djnz rdr_2
	ret                 ; last msnybble ends up in a lsnybble

;----------------------------------------------------------------------

patch_jp:
	ld (hl), 0C3h       ; JP instruction
	inc hl
	ld (hl), c
	inc hl
	ld (hl), b
	ld a, 6             ; bump to next, assuming this was RST n

add_hl_a:
	add a, l
	ld l, a
	ret nc
	inc h
	ret

bzero:
	xor a
memset:
	ld (hl), a
	ld d, h
	ld e, l
	inc de
	dec bc          ; length always 2 or more
	ldir
	ret

delay_100:
	ld a, 10
delay:              ; 17 reg A * 10usec, ballpark only, 4MHz XTAL
	dec a           ;  4 not quite correct adjustment for overhead
	ret z           ; 11 or 5
delay_1:
	push af         ; 11
	pop af          ; 10
	sub 1           ;  7
	jr nz, delay_1  ; 12 or 7
	ret             ; 10

jp_hl:
	jp (hl)

hl_using_bios_stack:
	push hl         ; if sp was 0 (empty bios stack), it now isn't
	ld hl, (-end) & 0FFFFh
	add hl, sp
	ret c           ; already. just go.

	pop hl
	ld (spsave), sp
	ld sp, 0
	call jp_hl

	ld sp, (spsave) ; back to user stack
	ret

;----------------------------------------------------------------------

home:
	ld bc, 0
settrk:
	ld (track), bc      ; 16 bits, includes partition offset
	ret

sectran:                ; bc is 0 ... spt-1
	ld h, b
	ld l, c             ; pass it back, keep zero-based
	ret

setsec:
	ld a, c             ; bc is 0 ... spt-1
	ld (sector), a      ; 8 bits XXX none of the listed spt:s is > 256
	ret

setbuf:
	ld (rwaddr), bc
	ret

seldsk:
	ld a, c
seldsk_a:
	ld hl, 0
	cp NDISK
	ret nc            ; NULL, if unknown
	add a, a
	add a, a
	add a, a          ; 4 words per index
	ld hl, dsktab     ;  pread, pwrite, dph, reserved
	call add_hl_a
	ld de, rwfunc     ; copy read and write functions
	ld bc, 4
	ldir
	ld e, (hl)        ; dph next
	inc hl
	ld d, (hl)
	ex de, hl         ; NULL, if missing
	ret

read:
	ld hl, (rwfunc + 0)
	jp hl_using_bios_stack

write:
	ld hl, (rwfunc + 2)
	jp hl_using_bios_stack

rw_fail:
ronly_fail:           ; XXX anything nz is error to bdos
	ld a, 1
	ret

;----------------------------------------------------------------------
; memory banks are 32kB each
; sector is 128 bytes
; spt is 256
; 256 banks, half rom, half ram, 4+4 MB
; dpb sizes/offsets handle reserved areas.
; track is one bank
; TRACK:SECTOR is linear 16:8 bits, laid out as 3 contiguous bytes
;
; on read/write entry/exit, ram bank 0 (standard) is assumed active.
;
; XXX might copy bytewise directly to dest ?
; XXX double LDIR vs much slower ld, out, ld, out loop ?

mem_wr_setup:
	ld de, (sectorx)      ; D sector, E 0
	srl d
	rr e
	ld a, (track)         ; A:DE, bank and offset

	ld hl, (rwaddr)       ; HL, transfer location in TPA+system
	ret

mem_read:
	call mem_wr_setup
	ex de, hl             ; A:HL is source bank:offset, DE rwaddr
	bit 7, d
	jr nz, shovel         ; destination in upper ram, direct copy works

	ld de, tmpbuf
	call shovel           ; get it to upper ram first

	ld hl, tmpbuf
	ld de, (rwaddr)       ; final destination
	call ldir_sector
	xor a
	ret

mem_write:
	call mem_wr_setup     ; A:DE banked location and HL rwaddr
	bit 7, h
	jr nz, mem_wr_1       ; source in upper ram, need not copy

	push de               ; keep the bank offset
	ld de, tmpbuf         ; is in high ram
	call ldir_sector
	pop de
	ld hl, tmpbuf         ; replace source ptr
mem_wr_1:
	bit 7, a
	jr z, ee_write        ; flash/eeprom banks 0...7F need more than ...

shovel:                   ; bank in a, offset in de/hl, buffer in hl/de
	call iff_disable      ; assuming 4MHz XTAL, interrupts are disabled for
	call bank_set         ; about, 700 usec XXX IM2 and stackswap, di not needed
	call ldir_sector
	call bank_std         ; back to TPA
	call iff_restore
	xor a
	ret                   ; a 0 = success

ldir_sector:
	ld bc, 128
	ldir                  ; or maybe rept 128 ldi endm
	ret

;----------------------------------------------------------------------
; XXX only 29C010 etc with 128 byte sector
; XXX cpu /WR rewire to chip /WE or appropriate other pin

ee_write:
	push af            ; remember bank

	call iff_disable   ; 700 usec or so XXX IM2 and stackswap, di not needed
	call bank_set      ; eeprom bank visible

	ld a, 0AAh
	ld (5555h), a
	ld a, 055h
	ld (2AAAh), a
	ld a, 0A0h         ; program a sector
	ld (5555h), a

	call ldir_sector   ; one sector
	call bank_std      ; back to TPA
	call iff_restore

	; wait completion, allowing interrupts between checks
	; load window closes in 150usec
	; programming completes in 10msec

	pop bc             ; bank in b
	dec hl
	dec de             ; back over to last LD

	ld c, 200          ; timeout in no less than 20 msec

	call delay_100     ; usec, maybe extra from interrupts

byte_poll:
	call delay_100

	ld a, b
	call peek          ; a = (a:de)
	sub (hl)
	ret z              ; a 0 = success

	dec c
	jr nz, byte_poll

	jp rw_fail

; id mode was 5555 <- AA
;             2AAA <- 55
;             5555 <- 90
; 10 msec
; id in 0 and 1
; leave id mode 5555 <- AA
;               2AAA <- 55
;               5555 <- F0
; 10 msec
;
; Flash magic was
;  5555 <- AA
;  2AAA <- 55
;  5555 <- 80
;  5555 <- AA
;  2AAA <- 55
;  addr <- 30 sector erase at addr, poll addr for FF
;
;  5555 <- AA
;  2AAA <- 55
;  5555 <- A0
;  addr <- byte byte program, poll addr for byte
;
; F040 64kB sectors
; F020 ?    sectors
; F010 16kB sectors
;----------------------------------------------------------------------
; CS does not sink in without 1 clock
; also, "accepted" write needs 8 clocks to start

sd_select:
	call sd_fini

	or SD_CS            ; select, still idle data
	out (RTC), a
	ret                 ; af b trashed

sd_fini:
	ld b, 17            ; clock low high low high ... low, 8 pulses
sd_wiggle:
	ld a, SD_DOUT       ; unselected, idle data
L1:
	out (RTC), a
	xor SD_CLK
	djnz L1
	ret                 ; af b trashed

sd_put:                 ; byte from a
	ld c, a
	ld b, 8
L3:
	ld a, SD_CS >> 1    ; 7 msbits
	rl c
	rla                 ; SD_DOUT is RTC.0
	out (RTC), a        ; clock is low
	or SD_CLK
	out (RTC), a        ; rising clock edge
	and ~SD_CLK
	out (RTC), a        ; leave with clock low
	djnz L3
	ret                 ; af bc trashed

sd_get:                 ; byte to a
	ld b, 8
L2:
	in a, (RTC)
	rla                 ; SD_DIN is RTC.7
	rl c
	ld a, SD_CS | SD_DOUT | SD_CLK
	out (RTC), a
	and ~SD_CLK
	out (RTC), a
	djnz L2

	ld a, c
	ret                 ; af bc trashed

; command in a, includes fixed msbits
; arg32 in dehl
; z return, if R1 (in a) is 0

sd_command_no_arg:
	ld hl, 0

sd_command_word_arg:  ; fits in HL
	ld de, 0

sd_command:           ; command in a, dword arg in dehl

	call sd_put       ; command includes fixed 01 startbits

	ld a, d           ; arg
	call sd_put
	ld a, e
	call sd_put
	ld a, h
	call sd_put
	ld a, l
	call sd_put

	ld a, 095h        ; crc7 only valid for initial CMD0
	call sd_put       ; DOUT ends up idle because of the stopbit

	ld hl, 20000      ; XXX timeout
L4:
	call sd_get       ; R1 response, when msbit is clear
	or a              ; zero and sign valid
	ret p             ; R1 in a. z if ok

	dec hl
	bit 7, h          ; until HL wraps negative
	jr z, L4
	ret               ; 0x80 | ? in a, nz

; command response R1
; 0x00 ok
; or bitfield
; 0x01 idle state
; 0x02 erase reset
; 0x04 illegal command
; 0x08 command crc error
; 0x10 erase sequence error
; 0x20 address error
; 0x40 parameter error
; 0x80 timeout (other bits meaningless)
;
; packet token
; 0xFF none yet
; 0xFE ok
; or bitfield
; 0x01 error
; 0x02 controller error
; 0x04 media ecc failed
; 0x08 out of range
; 0x10 card is locked

sd_wait_token:
	ld hl, 20000      ; XXX timeout
L5:
	call sd_get       ; token is first non-FF
	cp 0FFh
	ret nz            ; token in a

	dec hl
	bit 7, h          ; until HL wraps negative
	jr z, L5
	ret               ; FF in a

sd_wait_busy:
	ld hl, 20000      ; XXX timeout
L6:
	call sd_get       ; 8 clocks. data output activates after 1 clock
	inc a             ; FF is not busy
	ret z             ; z, ok

	dec hl            ; else busy or something
	bit 7, h          ; until HL wraps negative
	jr z, L6
	ret               ; nz, timeout

CMD0   equ 040h |  0 ; resets the card
CMD9   equ 040h |  9 ; read CSD
CMD10  equ 040h | 10 ; read CID
CMD16  equ 040h | 16 ; set R/W block
CMD17  equ 040h | 17 ; read block
CMD24  equ 040h | 24 ; write block
CMD55  equ 040h | 55 ; next command is ACMDxx
ACMD41 equ 040h | 41 ; send host capacity support, init card

; byte offset in dehl
; z return if ok, or error code in a

sd_write_block:
	ld a, CMD24           ; WRITE_BLOCK
	call sd_command       ; dehl byteaddress
	ret nz                ; not "ok"

	ld a, 0FEh            ; packet start token
	call sd_put

	ld hl, (rwaddr)
	ld e, 128             ; loop counter
L8:
	ld a, (hl)            ; data
	inc hl
	call sd_put
	dec e
	jr nz, L8             ; do the sector

	ld a, 0FFh
	call sd_put           ; crc16
	ld a, 0FFh
	call sd_put           ; XXX not used

	; xxx0___1
	;     010   accepted
	;     101   crc error
	;     110   write error XXX is not reporting _this_ block.

	call sd_wait_token    ; data response or FF if timed out
	and 01Fh
	cp  005h              ; "data accepted" ?
	ret

	; write will (?) really start only after 8 more clocks.
	; unselect does an extra get, so not a problem.

;	cp  00Bh              ; "transmission crc error" ?
;	jr z, retry           ; do it again

; byte offset in dehl
; z return if ok

sd_read_block:
	ld a, CMD17           ; READ_SINGLE_BLOCK
	call sd_command       ; dehl byteaddress
	ret nz                ; not "ok"

	call sd_wait_token    ; packet start or FF if timed out
	cp 0FEh
	ret nz                ;  or error

	ld hl, (rwaddr)
	ld e, 128             ; loop counter
L9:
	call sd_get           ; data
	ld (hl), a
	inc hl
	dec e
	jr nz, L9             ; do the sector

	call sd_get           ; crc16
	ld h, a
	call sd_get           ;  in HL
	ld l, a               ; XXX discarded

	xor a                 ; zero, no carry
	ret

sd_setup:
	call sd_select
	call sd_wait_busy     ; nz if timed out
	ret nz

	; byte address to dehl from TRACK, SECTOR
	; when spt 256 and sector 128

	ld hl, (sectorx)  ; sector to h, zero to l
	ld de, (track)    ; dehl (TRACK << 16) | (SECTOR << 8) | 0

	IF 0

	srl d
	rr e
	rr h
	rr l               ; dehl sector * 128

	ELSE

	; XXX XXX XXX XXX easy way out XXX 

	sla l
	rl h
	rl e
	rl d               ; XXX dehl sector * 512

	ENDIF

	xor a              ; z, ok
	ret

sd_write:
	call sd_setup          ; select, wait busy, calculate byte offset
	call z, sd_write_block
	jr sd_done

sd_read:
	call sd_setup
	call z, sd_read_block

sd_done:
	push af
	call sd_fini
	call rtc_unselect       ; /CS high, clock and dout low
	pop af
	jp nz, rw_fail

	xor a
	ret

;----------------------------------------------------------------------
; XXX resistor trick and DOUT kept low -> read in zeroes if no chip.
;
; DS1202 hardware layout, 7 bcd bytes
; {
;   second   0...59h
;   minute   0...59h
;   hour     0...23h
;   mday     1...31h
;   month    1...12h
;   wday     1...7h
;   year     0...99h
; }
;
; VIH 2.0v VIL 0.8v
; ce inactive 1us
; ce/clk setup 1us, clk/ce hold 60ns
; data setup 50ns, hold 70ns, delay 200ns
; clk low/high 250ns, freq 2MHz, rise/fall 500ns
; no delays, at least while CPU clk below 10MHz

rtc_begin:             ; command in c
rtc_out:               ; byte in c
	ld b, 8            ; bit count
L10:
	rrc c              ; LSbit goes first
	ld a, RTC_CE << 1  ; room for A.7 aka RTC_DOUT
	rra                ; A.7 RTC_DOUT valid
	out (RTC), a       ; data valid, chip enabled, clock low
	or RTC_CLK
	out (RTC), a       ; rising CLK, chip samples data
	djnz L10
	ret                ; byte still in c, clock high, last databit visible

rtc_in:
	ld a, RTC_CE | RTC_DIR    ; DOUT is 0 when reading
	out (RTC), a
	ld b, 8            ; bit count
L11:
	in a, (RTC)        ; A.0 is RTC_DIN
	rra
	rr c               ; LSbit comes first

	ld a, RTC_CE | RTC_DIR | RTC_CLK
	out (RTC), a
	and ~RTC_CLK
	out (RTC), a       ; falling CLK, chip changes data
	djnz L11
	ret                ; byte in c

rtc_write_enable:
	ld c, 8Eh          ; CONTROL WRITE
	call rtc_begin

	ld c, 00h          ; WRITE_PROTECT=0
rtc_fini:
	call rtc_out       ; fall thru to ...

rtc_unselect:
rtc_init:
	xor a
	out (RTC), a
	ret

rtc_write:             ; command in c, data in hl, length e
	call rtc_begin
L15:
	ld c, (hl)
	inc hl
	call rtc_out
	dec e
	jr nz, L15
	ret

; set/get clock/nvram

nvram_get:
	ld c, 0FFh         ; RAM BURST READ
	ld e, 24           ; 24 bytes
	jr L12

clock_get:
	ld c, 0BFh         ; CLOCK BURST READ
	ld e, 7            ; 7 bcd bytes
L12:
	call rtc_begin
L13:
	call rtc_in
	ld (hl), c
	inc hl
	dec e
	jr nz, L13
	jr rtc_unselect

; clock_halt   flag in bit 7 of hl->rtc_second and
; 12_hour_mode flag in bit 7 of hl->rtc_hour assumed to be clear

clock_set:
	call rtc_write_enable

	ld c, 0BEh         ; CLOCK BURST WRITE
	ld e, 7            ; 7 bcd bytes
	call rtc_write
	jr L14             ; CONTROL byte is next automatically

nvram_save:
	call rtc_write_enable

	ld c, 0FEh         ; RAM BURST WRITE
	ld e, 24           ; 24 bytes
	call rtc_write
	call rtc_unselect  ; position to CONTROL byte

	ld c, 8Eh          ; CONTROL WRITE
	call rtc_begin
L14:
	ld c, 80h          ; WRITE_PROTECT=1
	jr rtc_fini        ; out and unselect

;----------------------------------------------------------------------

; XXX interrupt cycle might affect BANK latch.
; XXX restore from cbank before ret ?

interrupt_ret:
	xor a
	ld (iff), a
	pop de
	pop bc
	pop af
	pop hl
	ld sp, (int_stack)
	ei
	ret

interrupt:
	ld (int_stack), sp
	ld sp, int_stack    ; XXX unavoidable, it seems
	push hl
	push af
	push bc
	push de
	ld a, 1
	ld (iff), a

check_iir:
	in a, (UART_IIR)
	rra
	jr c, interrupt_ret ; xxx1 is nothing

	rra
	jr c, L20           ; xx10 is thre or ls change
	rra
	jr nc, uart_modem_status

	;------------------
	; x100 rx data available (trigger level or fifo timeout)

	in a, (UART_DAT)
	ld hl, (rx_silo_ins)
	ld (hl), a
	inc l                 ; aligned 256 bytes
	ld (rx_silo_ins), hl

	; XXX ld (BIOS_004x+15), a  ; XXX cheap ^C checks

	jr check_iir          ; trigger level is 1, no need to test LSR.0
L20:
	rra
	jr c, uart_line_status

	;------------------
	; x010 tx holding reg empty (fifo empty)

	jr check_iir

	;------------------
	; x000 change in modem inputs XXX poor man's PIC ?
	; 0 DCTS, 1 DDSR, 2 TERI, 3 DDCD, 4 CTS, 5 DSR, 6 RI, 7 DCD

uart_modem_status:
	in a, (UART_MSR)

	bit 3, a                ; DCD edge ?
	call nz, bump_time

	jr check_iir

	;------------------
	; x110 parity/framing error or break
	; 0 DR, 1 OE, 2 PE, 3 FE, 4 BI, 5 THRE, 6 TEMT, 7 ERR_IN_FIFO

uart_line_status:
	in a, (UART_LSR)

	bit 4, a                ; BRK ?
	call nz, break_interrupt

	jr check_iir

; XXX and/or ?

break_interrupt:
	ld hl, conbaud
	inc (hl)
	call uart_init
	jp wboote

uart_init:
	ld hl, conbaud
	ld a, (hl)
	cp NBAUD
	jr c, L30
	xor a
	ld (hl), a
L30:
	add a, a             ; word index
	inc a                ; step over to baud table
	call add_hl_a        ; (hl) is brg divisor. word wide.

	ld a, 083h           ; 8N1, DLAB
	out (UART_LCR), a
	ld a, (hl)
	inc hl               ; to msbyte
	out (UART_DLL), a
	ld a, (hl)
	out (UART_DLH), a

	ld a, 003h           ; 8N1
	out (UART_LCR), a
	ld a, 000h           ; modem etc outputs high
	out (UART_MCR), a
	ld a, 000h           ; no interrupts enabled
	out (UART_IER), a

	ld a, 001h           ; enable fifos (trigger level 1)
	out (UART_FCR), a
	ld a, 003h           ; clear rx fifo
	out (UART_FCR), a
	ret

ttist:
	ld hl, (rx_silo_rem)
	ld a, (rx_silo_ins)
	sub l
	ret z                  ; not ready, z, 0
	ld a, 0FFh
	ret                    ; ready, nz, ff

ttin:
	call ttist
	jr z, ttin
	ld a, (hl)
	inc l
	ld (rx_silo_rem), hl
	ret                    ; XXX no meaningful flags

; XTAL -> 4040 -> DCD
;
; 4194304 / 16384 = 256 Hz
; 3686400 / 16384 = 225 Hz
; remember DI/EI in shovel

TIME_STEP  equ (10000h * 16384 + XTAL/2) / XTAL

TIME_ERROR equ XTAL / (10000h * 16384.0 / TIME_STEP) ; XXX 1000ppm

TIMEX equ BIOS_004x + 0  ; seconds, 16.16 format

bump_time:
	ld bc, TIME_STEP
	ld hl, (TIMEX+0)
	add hl, bc
	ld (TIMEX+0), hl
	ret nc               ; not a second yet

	ld hl, (TIMEX+2)
	inc hl
	ld (TIMEX+2), hl
	ret

;----------------------------------------------------------------------
; divisor = XIN / 16 / baud
;
; XIN might be 1.8432, 3.6864 or 4 MHz

BAUD macro bd
	dw (UART_XTAL + bd * 8) / (bd * 16)
	endm

conbaud:
	db 6        ;     select 1200 bauds XXX autobaud it, start 19200.
bauds:          ;     follows conbaud immediately
	BAUD   1200 ; 0   1201 if 4MHz XIN
	BAUD   1800 ; 1
	BAUD   2400 ; 2   2403
	BAUD   3600 ; 3
	BAUD   4800 ; 4   4807
	BAUD   7200 ; 5
	BAUD   9600 ; 6   9615
	BAUD  14400 ; 7
	BAUD  19200 ; 8  19230
	BAUD  28800 ; 9
	BAUD  31250 ; A  31250 (midi)
	BAUD  38400 ; B  35714
	BAUD  57600 ; C  62500
	BAUD 115200 ; D 125000

NBAUD equ ($ - bauds) / 2    ; count of words XXX forget the list, use algebra

rx_silo_ins: dw rx_silo
rx_silo_rem: dw rx_silo

;----------------------------------------------------------------------

NDISK equ 4

rwfunc: dw rw_fail, rw_fail

dsktab:
	dw mem_read, ronly_fail ; XXX mem_write
	dw dph_rom, 0           ; should always exist

	dw mem_read, mem_write
	dw 0, 0                 ; only exists if ram >= 128kB

	dw sd_read, sd_write
	dw 0, 0

	dw sd_read, sd_write    ; 2nd partition into SD
	dw 0, 0

dph_rom: dw 0, 0, 0, 0, dirbf, dpb_rom, chk_nil, all_rom
dph_ram: dw 0, 0, 0, 0, dirbf, dpb_ram, chk_nil, all_ram
dph_sd0: dw 0, 0, 0, 0, dirbf, dpb_sd0, chk_nil, all_sd0
dph_sd1: dw 0, 0, 0, 0, dirbf, dpb_sd1, chk_nil, all_sd1

; romdisk. 27512 or larger needed.
; XXX overwritten from bootrom offset 80h

nblks_rom equ 496    ; 1MB/2kB - 32kB (boot) XXX just to reserve all_rom
dpb_rom:
	dw 256           ; spt
	db 3, 7          ; bsh, blm. 1kB
	db 0             ; exm
	dw nblks_rom - 1 ; dsm
	dw 63            ; drm
	db 0C0h, 0       ; al0/1
	dw 0             ; cks
	dw 1             ; off XXX skip 32kB, boot area

; ramdisk. 128kB sram or larger needed.
; initial values for 256kB with 1kB blocks
; XXX patched during cold boot probe

nblks_ram equ 480    ; 1MB/2kB - 64kB (TPA+system) XXX just to reserve all_ram
dpb_ram:
	dw 256           ; spt
	db 3, 7          ; bsh, blm. 1kB
	db 0             ; exm
	dw 0             ; dsm
	dw 63            ; drm
	db 0C0h, 0       ; al0/1
	dw 0             ; cks
	dw 0             ; off XXX patched to skip into first ram bank

; SD. 8MB or larger needed.
; two partitions,
; former has one reserved track,
; latter can be moved around 1...nslot-1 to reach 2GB

nblks_sd0 equ 2040   ; 8MB - 32kB
dpb_sd0:
	dw 256           ; spt
	db 5, 31         ; bsh, blm. 4kB
	db 1             ; exm
	dw nblks_sd0 - 1 ; dsm
	dw 511           ; drm
	db 0F0h, 0       ; al0/1
	dw 0             ; cks
	dw 1             ; off XXX skip 32kB

nblks_sd1 equ 2048   ; 8MB exactly
dpb_sd1:
	dw 256           ; spt
	db 5, 31         ; bsh, blm. 4kB
	db 1             ; exm
	dw nblks_sd1 - 1 ; dsm
	dw 511           ; drm
	db 0F0h, 0       ; al0/1
	dw 0             ; cks
	dw 256           ; off XXX second 8MB slot initially

bootdev:   db 0   ; 0 rom
batt_err:  db 0   ; DS1210 error flag

cbank:     db 0   ; current bank, mirror of BANK latch
ram_bank0: db 0   ; probed by cold boot

iff:       db 1   ; nesting count of interrupt disables

;----------------------------------------------------------------------
	IF 0

	ccp.bin load with the help of bdos
	must play tricks with stack (bios -> bdos -> bios)

	ld c, 13              ; reset disk
	call BDOS

	ld c, 35              ; get filesize
	ld de, fcb
	call BDOS

fcb:
	db 1, "CCP     BIN", 0, 0, 0, 0
	db 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	db 0
fcb_nsect:
	dw 0
	db 0

	ENDIF
;----------------------------------------------------------------------
; from rom bootstrap. nothing has been done yet,
; rom bank 0 visible.
;
; everything below will be used for buffers and such
; after cold boot has executed.
;
; no initialisation of bss variables here XXX it WILL bite...

bss:
boot:
	ld sp, 0             ; above bss
	di
	im 1

	ld (batt_err), a     ; DS1210 low batt indication
	ld a, b
	ld (bootdev), a      ; 0 rom, no others

	ld a, 0FFh
	ld i, a              ; XXX move refresh cycles out of bank window

	ld a, 0C3h           ; JP instruction
	ld hl, reload_bios

	ld (boote+1), hl     ; boot: vanishes when done.

	; standard bank is still junk

	ld c, BANK
	ld e, BANK_TPA
	out (c), e           ; ram bank 0 visible, NMI will crash ...

	ld (67h), hl
	ld (66h), a          ; NMI ... ok again.

	ld (39h), hl
	ld (38h), a          ; RST 7

	ld (01h), hl
	ld (00h), a          ; RST 0

	call bank_std        ; almost a no-op, but keep books

	ld l, 10111100b      ; IOBYTE LPT: UP2: UR2: TTY:
	ld h, 00h            ; CDISK  A0
	ld (IOBYTE), hl

	call ppi_init        ; output pins to inactive
	call rtc_init        ; output pins to zero
	call uart_init       ; IER left zero

	ld hl, banner_1
	call boomsg          ; XXX count cycles until UART TEMT, gives XIN

	; romdisk

	ld a, 0              ; allow patching XXX shuffled ?
	call bank_set

	ld hl, ROM_DPB       ; XXX patch offset to newest generation ? XXX
	ld de, dpb_rom
	ld bc, 15
	ldir

	call bank_std

	ld hl, banner_2
	call boomsg

	; ramdisk
	;
	; probe ram. min 2, max 128 banks, 64kB ... 4MB total ram size.
	; write 80...FF, result is repeating FC..FF, if 4 banks.
	;
	; ram banks are still garbage, except TPA (just
	; restarts etc are valid) and system bank obviously.

	ld hl, TPA           ; 0100 or 8100, a safe byte to trash

	ld de, tmpbuf        ; XXX bss, but safely above boot routine
	ld a, 0
probe_0:                 ; banks FF to 80
	dec a
	call bank_set
	ldi                  ; save the probe byte of each bank
	dec hl               ; keep hl at probe byte
	cp RAM_ENB
	jr nz, probe_0

probe_1:                 ; banks 80 to FF
	call bank_set
	ld (hl), a           ; mark it
	inc a
	jr nz, probe_1

probe_2:                 ; banks FF to 80
	dec a
	call bank_set
	cp (hl)              ; check mark
	jr nz, probe_3
	cp RAM_ENB
	jr nz, probe_2

	dec a                ; 128 banks. whoa.

probe_3:
	inc a                ; first valid ram bank XXX should be power of 2
	ld (ram_bank0), a

	ex de, hl            ; probe byte to de, end of saved bytes to hl
	dec hl               ; last saved byte
	ld a, RAM_ENB
probe_4:                 ; banks 80 to FF
	call bank_set
	ldd                  ; restore original data
	inc de               ; keep de at probe byte
	inc a
	jr nz, probe_4

	call bank_std

	; XXX nblks_ram is not the theoretical max, but 1MB
	; XXX if reserving banks for other stuff,
	; XXX carve out from the end (SUB N now) XXX maybe from the start ?
	;
	;  128kB   2 banks  3,  7, 0,   63,  63, C0 00   1k block
	;  256kB   6 banks  3,  7, 0,  191,  63, C0 00   1k block
	;          7 banks  3,  7, 0,  223,  63, C0 00   1k block
	;          8 banks  3,  7, 0,  255,  63, C0 00   1k block

	;          9 banks  4, 15, 1,  143, 127, C0 00   2k block
	;         10 banks  4, 15, 1,  159, 127, C0 00   2k block
	;         11 banks  4, 15, 1,  175, 127, C0 00   2k block
	;         12 banks  4, 15, 1,  191, 127, C0 00   2k block
	;         13 banks  4, 15, 1,  207, 127, C0 00   2k block
	;  512kB  14 banks  4, 15, 1,  223, 127, C0 00   2k block
	;         15 banks  4, 15, 1,  239, 127, C0 00   2k block
	;         16 banks  4, 15, 1,  255, 127, C0 00   2k block

	;         17 banks  4, 15, 0,  271, 127, C0 00   2k block
	;         18 banks  4, 15, 0,  287, 127, C0 00   2k block
	;         19 banks  4, 15, 0,  303, 127, C0 00   2k block
	;         20 banks  4, 15, 0,  319, 127, C0 00   2k block
	;         21 banks  4, 15, 0,  335, 127, C0 00   2k block
	;         22 banks  4, 15, 0,  351, 127, C0 00   2k block
	;         23 banks  4, 15, 0,  367, 127, C0 00   2k block
	;         24 banks  4, 15, 0,  383, 127, C0 00   2k block
	;         25 banks  4, 15, 0,  399, 127, C0 00   2k block
	; 1024kB  30 banks  4, 15, 0,  479, 127, C0 00   2k block
	; 2048kB  62 banks  4, 15, 0,  991, 127, C0 00   2k block
	; 4096kB 126 banks  4, 15, 0, 2015, 127, C0 00   2k block

	ld a, (ram_bank0)
	ld b, a              ; remember
	neg                  ; 1st bank into bank count
	sub 2                ; how many banks for ramdisk ?
	jr z, no_ramdisk     ; only 64kB, TPA and system banks use it all

	ld ix, dpb_ram
	ld (ix+13), b        ; off. start track equals start bank

	ld l, a
	ld h, 0              ; bank count to 1k blocks
	add hl, hl
	add hl, hl
	add hl, hl
	add hl, hl
	add hl, hl

	cp 9
	jr c, can_use_1k_blocks
	cp 17
	jr nc, exm_ok
	inc (ix+4)           ; exm
exm_ok:
	inc (ix+2)           ; bsh
	scf
	rl (ix+3)            ; bsm
	scf
	rl (ix+7)            ; drm doubles
	rl (ix+8)
	srl h                ; blockcount halved
	rl l
can_use_1k_blocks:

	dec hl               ; dsm - 1
	ld (ix+5), l
	ld (ix+6), h

	; whew.

	; initialise ramdisk directory XXX keep if battery backed

	IF 0

	ld a, (ram_bank0)
	call bank_set        ; first ram bank visible

	ld a, 0E5h           ; just fill one full 32kB bank,
	ld hl, 0             ; 1024 dirents.
	ld bc, 8000h
	call memset

	ENDIF

	ld hl, dph_ram
	ld (dsktab+12), hl   ; exists. selectable.
no_ramdisk:

	call bank_std

	ld hl, banner_3
	call boomsg

	; probe SD

	call sd_init
	jr nz, no_sd

	ld hl, dph_sd0
	ld (dsktab+20), hl

	ld hl, dph_sd1
	ld (dsktab+28), hl
no_sd:

	ld hl, banner_4
	call boomsg

	; bzero TPA and bss, then wboote

	ld hl, wboote
	push hl

	ld hl, TPA
	ld bc, RAM_CCP - TPA
	call bzero

	ld hl, bss
	ld bc, end - bss
	jp bzero             ; returns to wboote

;----------------------------------------------------------------------
; SD probe, disappears after cold boot

sd_init:
	ld b, 255
	call sd_wiggle
	call sd_select      ; ignore busy condition
	call sd_init_1
	jp sd_done

; z return if ok
; idle, ready, size, set block size

sd_init_1:
	ld a, CMD0            ; GO_IDLE_STATE
	call sd_command_no_arg
	cp 001h
	ret nz                ; not "idle"

sd_notready:
	ld a, CMD55           ; APP_CMD
	call sd_command_no_arg
	and ~001h
	ret nz                ; not "idle" nor "ok"

	ld a, ACMD41          ; SD_SEND_OP_COND. arg 0x40000000 is HCS
	call sd_command_no_arg
	cp 001h
	jr z, sd_notready     ; wait, while idle
	or a
	ret nz                ; not ok

	IF 0 ;;;;;;;;;;;;;;;;;;

	ld a, CMD9            ; SEND_CSD
	call sd_command_no_arg
	ret nz                ; not ok

	call sd_wait_token    ; packet start or FF if timed out
	cp 0FEh
	ret nz                ;  or error

	ld hl, tmpbuf         ; XXX
	ld e, 16              ; not counting crc16
L7:
	call sd_get
	ld (hl), a
	inc hl
	dec e
	jr nz, L7             ; do 16 bytes

	call sd_get           ; crc16
	call sd_get           ; discarded

	; XXX calculate SD size from CSD bits, in tmpbuf
	; XXX might not need it at all.

	ENDIF ;;;;;;;;;;;;;;;;;

	ld a, CMD16           ; SET_BLOCKLEN
	ld hl, 128
	call sd_command_word_arg
	ret                   ; z or nz

;----------------------------------------------------------------------

boomsg_1:
	ld c, a
	push hl
	call ttyout
	pop hl
	inc hl
boomsg:
	ld a, (hl)
	or a
	jr nz, boomsg_1
	ret

banner_1:  db 0Dh, 0Ah
           db "zanzibar", 0Dh, 0Ah, 0
banner_2:  db "ram...", 0
banner_3:  db " sd...", 0
banner_4:  db " done.", 0Dh, 0Ah, 0

;----------------------------------------------------------------------
; zeroed after cold boot has done it's job.
; NO DB or DW any more !

	org bss

spsave:  ds 2    ; user stack saved during read/write
rwaddr:  ds 2    ; aka dmaad

sectorx: ds 1    ; always zero
sector:  ds 1    ;  this layout is convenient
track:   ds 2    ;   in cases where spt is 256. bits 16:8 contiguous

rdr_buf: ds 7    ; UR2: reads "yymmddHHMMSS\x1A"

dirbf:   ds 128

all_rom: ds nblks_rom / 8 ; one bit per block on disk
all_ram: ds nblks_ram / 8
all_sd0: ds nblks_sd0 / 8
all_sd1: ds nblks_sd1 / 8
chk_nil: ds 0

	ds 2*16      ; 16 words
int_stack:
	ds 2         ; and old sp

	align 256

rx_silo: ds 256  ; aligned 256 bytes
rx_silo_end:

tmpbuf:  ds 128  ; XXX

;----------------------------------------------------------------------

end: ; rest is stack

