
	include "cpm.inc"
	org TPA

	jp start

; 8243 control pins. another 8243 has CS inverted

PROG equ 1
CS   eq  2

static void
ctrl(u_char x)
{
	outb(LPT, x);
	inb(LPT);
}

; 8 4-bit ports in a pair of 8243: 0..3 in first, 4..7 in second.
; 5 ports for address, including the possible /WE etc pins.
; 2 ports for databus.
; 1 port for CE/OE/VppXXX enables.

aports:
	db 0, 2, 4, 6, 3

LODAT equ 1
HIDAT equ 5

CE     equ 8 ; CE and OE inverted for convenience
OE     equ 4
VppA15 equ 2 ; Vpp into A15 pin, overrides address bit
VppOE  equ 1 ;  into OE/Vpp, overrides OE bit

enables:            ; from register a
	ld c, 7         ; port 7
	xor CE | OE
	jp put

; some common address bits. note WE is an address pin, not an 'enable'

WE  equ 1 << 14
PGM equ 1 << 14
Vpp equ 1 << 15

sector_size:   ; in bytes, or 0
	dw 0

amask:         ; 0x0FFF
	dw 0

write_enable:  ; 1 << 14
	dw 0

must_be_one:   ; 1 << 15
	dw 0

quirks:
	db 0

quirk_WE    equ 1 ; /WE controlled writes on 2817
quirk_A14   equ 2 ; A14 leg on 28256
quirk_4802  equ 4 ; 0x7F0 to 0x7FF fails verify on 2kB timekeeper

write_fcn:
	dw 0

static void
progress(u_long x)
{
	if ((x & 0x3FF) == 0)
		fprintf(stderr, "\r%05X ", x);
}

static void
delay_ms(int msec)
{
}

static void
delay_us(int usec)
{
}

; P2 pins

static u_char
din()
{
	return (inb(LPT + 2) & 0xF ^ 0xB);
}

static void
dout(u_char x)
{
	outb(LPT + 2, x & 0xF ^ 0xB);
}

static void
put(u_char port, u_char nybb)
{
	u_char cs;

	cs = (port & 4) ? CS : 0;
	port &= 3;

	ctrl(cs | PROG);
	dout(port | 4);   /* P23 P22 == 0 1 == Write */
	ctrl(cs);
	dout(nybb);
	ctrl(cs | PROG);
}

static u_char
get(u_char port)
{
	u_char cs, nybb;

	cs = (port & 4) ? CS : 0;
	port &= 3;

	ctrl(cs | PROG);
	dout(port | 0);   /* P23 P22 == 0 0 == Read */
	ctrl(cs);        /* series resistors prevent HCF while ...  */
	xout(~0);        /* ... this has not happened. */
	nybb = din();
	ctrl(cs | PROG);

	return (nybb);
}

static void
address(u_long addr)
{
	static u_long old = ~0;

	u_long was = old;
	int port;

	if (old == ~0)
		was = ~addr;

	old = addr;

	/* nybbles to 5 ports numbered 0..4 */

	for (port = 0; port <= 4; ++port, addr >>= 4, was >>= 4)
		if ((addr ^ was) & 0xF)
			put(aports[port], addr);
}

static int write_mode = 1;

static u_char
rd_databus()
{
	write_mode = 0;

	return (get(LODAT) | (get(HIDAT) << 4));
}

static void
wr_databus(u_char data)
{
	write_mode = 1;

	put(LODAT, data);
	put(HIDAT, data >> 4);
}

static u_long
fix_address(u_long addr)
{
	if (quirks & quirk_A14)
		addr += addr & (1<<14); /* 28256 A14 at pin 1, normally A15 */

	return (addr | must_be_one);
}

/* rd and wr not used by prom write functions */

static u_char
rd(u_long addr)
{
	u_char data;

	/* databus is Hi-Z after a junk Read */

	if (write_mode)
		(void) rd_databus();

	address(fix_address(addr) | write_enable);
	enables(CE | OE);
	data = rd_databus();
	enables(0);

	return (data);
}

static void
wr(u_long addr, u_char data)
{
	if (quirks & quirk_WE)
		address(fix_address(addr) | write_enable);

	address(fix_address(addr) & ~write_enable);
	wr_databus(data);
	enables(CE);

	if (quirks & quirk_WE)
		address(fix_address(addr) | write_enable);

	enables(0);
}




/* generic RAM; just write it */

static char *
w_ram(u_long x, u_long end)
{
	for ( ; x < end; ++x) {
		u_char data = file[x];

		wr(x, data);
		if (rd(x) ^ data)
			return ("Error");
	}
	return (0);
}


/* generic EEPROM; write and wait program complete */

static char *
w_eeprom(u_long x, u_long end)
{
	for ( ; x < end; ++x) {
		u_char data = file[x];

		if (rd(x) ^ data) {
			int timo = 100000;

			wr(x, data);
			while (rd(x) ^ data)
				if (--timo == 0)
					return ("Error");
		}
	}
	return (0);
}


static int
sector_ok(u_long addr)
{
	u_long end = addr + sector_size;

	do {
		if (rd(addr) ^ file[addr])
			return (0);
	} while (++addr < end);

	return (1);
}

static void
magic_sequence(u_char cmd)
{
	wr(0x5555 & amask, 0xAA);
	wr(0x2AAA & amask, 0x55);
	wr(0x5555 & amask, cmd);
}

static void
product_id()
{
	static struct list {
		u_char code;
		char *name;
	}
		devices_amd[] = {
			{ 0x01, "AMD",       },
			{ 0xA4, "Am29F040"   },
			{ }
		},
		devices_atmel[] = {
			{ 0x1F, "Atmel",     },
			{ 0xDC, "AT29C256"   },
			{ 0x5D, "AT29C512"   },
			{ 0xD5, "AT29C010A"  },
			{ 0xDA, "AT29C020"   },
			{ 0xA4, "AT29C040A"  },
			{ }
		},
		devices_st[] = {
			{ 0x20, "ST",        },
			{ 0xE2, "M29F040B"   },
			{ 0xE3, "M29W040B"   },
			{ }
		},
		devices_ti[] = {
			{ 0x97, "TI",        },
			{ 0xF1, "TMS29F256", },
			{ }
		},
		*devices[] = {
			devices_amd,
			devices_atmel,
			devices_st,
			devices_ti,
			NULL
		};
	struct list **pp, *p;
	u_char mc, dc;

	magic_sequence(0x90); /* Autoselect */
	delay_ms(10);
	mc = rd(0x0000);
	dc = rd(0x0001);

	magic_sequence(0xF0); /* Read/Reset */
	delay_ms(10);

	for (pp = devices; p = *pp; ++pp)
		if (p->code == mc) {
			p++;
			for ( ; p->name; ++p)
				if (p->code == dc)
					break;
			break;
		}

	if (p)
		fprintf(stderr, "%s", (*pp)->name);
	else
		fprintf(stderr, "manufacturer %02X", mc);

	fprintf(stderr, ", ");

	if (p && p->name)
		fprintf(stderr, "%s", p->name);
	else
		fprintf(stderr, "device %02X", dc);

	fprintf(stderr, "\n");
}

/* flash; erase a sector and write the bytes one by one */

static char *
w_flash(u_long start, u_long end)
{
	u_long x, y;
	int timo;

	for (x = start; x < end; x += sector_size) {

		if (sector_ok(x))
			continue;

		timo = 100000;
		magic_sequence(0x80); /* sector erase at x */
		wr(0x5555, 0xAA);
		wr(0x2AAA, 0x55);
		wr(x,      0x30);
		while (rd(x) ^ 0xFF)
			if (--timo == 0)
				return ("Timeout in erase");

		for (y = x; y < x + sector_size; ++y) {
			u_char data = file[y];

			if (rd(y) ^ data) {

				timo = 1000;
				magic_sequence(0xA0);
				wr(y, data);
				while (rd(y) ^ data)
					if (--timo == 0)
						return ("Timeout in program");
			}
		}
	}
	return (0);
}

/* paged eeprom or autoerasing sectored flash */

static char *
w_autoerase(u_long start, u_long end)
{
	u_long x, y;
	int timo;

	for (x = start; x < end; x += sector_size) {

		if (sector_ok(x))
			continue;

		/* load one sector/page/whatever __uninterruptibly__ */

		disable_intr(); /* XXX a page fault will foul things up */

		magic_sequence(0xA0);

		for (y = x; y < x + sector_size; ++y)
			wr(y, file[y]);

		enable_intr();

		/* poll last write and verify all */

		timo = 1000;
		--y;
		while (rd(y) ^ file[y])
			if (--timo == 0)
				return ("Timeout");

		if (! sector_ok(x))
			return ("Failed");
	}
	return (0);
}


static u_char
vfy(u_char enb)
{
	u_char data;

	if (write_mode)
		(void) rd_databus();

	enables(enb | OE);
	data = rd_databus();
	enables(enb);

	return (data);
}

/* PGM programs, OE verifies */

static char *
w_prom_PGM(u_long start, u_long end)
{
	u_char enb = CE | VppA15;    /* CE stays low, Vpp stays on */
	u_long x;
	int n;

	address(start | Vpp | PGM);  /* pin 1 and PGM at TTL high */
	enables(enb);                /* pin 1 at Vpp */
	delay_ms(10);

	for (x = start; x < end; ++x) {
		u_char diff, data = file[x];

		progress(x);

		address(x | Vpp | PGM);

		diff = vfy(enb) ^ data;
		if (diff) {
			if (diff & data)
				return ("Erasure required");

			n = 0;
			do {
				if (++n == 25)
					return ("Error");

				wr_databus(data);
				address(x | Vpp);
				delay_us(1000);
				address(x | Vpp | PGM);

			} while (vfy(enb) ^ data);

			wr_databus(data);
			address(x | Vpp);
			delay_us(1000 * 3 * n);
			address(x | Vpp | PGM);
		}
	}
	return (0);
}

static void
program(u_char enb, u_char data, int usec)
{
	wr_databus(data);
	enables(enb | CE);
	delay_us(usec);
	enables(enb);
}

/* CE programs, OE verifies */

static char *
w_prom_256(u_long start, u_long end)
{
	u_char enb = VppA15;    /* Vpp stays on */
	u_long x;
	int n;

	address(start | Vpp);   /* pin 1 at TTL high */
	enables(enb);           /* pin 1 at Vpp */
	delay_ms(10);

	for (x = start; x < end; ++x) {
		u_char diff, data = file[x];

		progress(x);

		address(x | Vpp);

		diff = vfy(enb) ^ data;
		if (diff) {
			if (diff & data)
				return ("Erasure required");

			n = 0;
			do {
				if (++n == 25)
					return ("Error");

				program(enb, data, 1000);

			} while (vfy(enb) ^ data);

			program(enb, data, 1000 * 3 * n);
		}
	}
	return (0);
}

/* CE programs, no verify-mode */

static char *
w_prom_512(u_long start, u_long end)
{
	u_long x, todo;
	u_char n[0x10000];

	bzero(n, sizeof(n));

	todo = 0;
	for (x = start; x < end; ++x) {
		u_char diff = rd(x) ^ file[x];

		if (diff) {
			if (diff & file[x])
				return ("Erasure required");
			++todo;
			n[x] = 0x81;
		}
	}
	if (! todo)
		return;

	while (todo) {
		enables(VppOE); /* OE was TTL high, now Vpp */
		delay_ms(10);

		for (x = start; x < end; ++x) {
			progress(x);

			if (n[x] & 0x80) {
				address(x);
				program(VppOE, file[x], 500);
			}
		}
		enables(0);        /* OE back to TTL high */
		delay_ms(10);

		for (x = start; x < end; ++x)
			if (n[x] & 0x80) {
				if (rd(x) == file[x]) {
					--todo;
					n[x] &= ~0x80;

				} else if (++n[x] == 0x80 + 25)
					return ("Error");
			}
	}
	fprintf(stderr, "\nOverprogramming\n");

	enables(VppOE); /* OE was TTL high, now Vpp */
	delay_ms(10);

	for (x = start; x < end; ++x) {
		progress(x);

		if (n[x]) {
			address(x);
			program(VppOE, file[x], 500 * 3 * n[x]);
		}
	}
	enables(0);        /* OE back to TTL high */
	delay_ms(10);

	return (0);
}

static char *
w_notyet(u_long start, u_long end)
{
	return ("Write unsupported");
}

static struct chiptype {
	char *name;
	void          *w_fcn;
	u_long                       kB, sector, we,    mbo,   quirks;
} chips[] = {
	{ "4802",      w_ram,         2,      0, 1<<11, 1<<13, },
	{ "6116",      w_ram,         2,      0, 1<<11, 1<<13, },
	{ "2816",      w_eeprom,      2,      0, 1<<11, 1<<13, },

	{ "2817",      w_eeprom,      2,      0, WE,    0,     quirk_WE },
	{ "9817",      w_eeprom,      2,      0, WE,    0,     quirk_WE },

	{ "6264",      w_ram,         8,      0, WE,    0,     },
	{ "2864",      w_eeprom,      8,      0, WE,    0,     },
	{ "X28C64",    w_autoerase,   8,     64, WE,    0,     },
	{ "2764",      w_prom_PGM,    8,      0, PGM,   Vpp,   },

	{ "62128",     w_ram,        16,      0, WE,    0,     },
	{ "27128",     w_prom_PGM,   16,      0, PGM,   Vpp,   },

	{ "62256",     w_ram,        32,      0, 0,     0,     },
	{ "28256",     w_eeprom,     32,     64, WE,    0,     quirk_A14 },
	{ "27256",     w_prom_256,   32,      0, 0,     Vpp,   },
	{ "AT29C256",  w_autoerase,  32,     64, 1<<19, 0,     }, /* XXX */

	{ "27512",     w_prom_512,   64,      0, 0,     0,     },
	{ "AT29C512",  w_autoerase,  64,    128, 1<<19, 0,     }, /* XXX */

	{ "AT29C010A", w_autoerase, 128,    128, 1<<19, 0,     },
	{ "AM29F010",  w_flash,     128,  16384, 1<<19, 0,     },

	{ "AT29C020",  w_autoerase, 256,    256, 1<<19, 0,     }, /* XXX */
	{ "29020",     w_notyet,    256,      1, 1<<19, 0,     },

	{ "AT29C040A", w_autoerase, 512,    256, 1<<19, 0,     }, /* XXX */
	{ "29F040",    w_flash,     512,  65536, 1<<19, 0,     },


	/* 29F040 playing a 27C256. /WE at pin 1 (Vpp), A15...18 grounded */

	{ "29F256",    w_flash,      32,  32768, Vpp,   0,     },

	/* 29F040 playing a 27C512. dangling wire /WE, A16 to pin1, A15,17,18 Vcc */

	{ "29F512",    w_flash,      64,  32768, 1<<19, 0,     },

	{ } /* End */
};

static int
lookup_chip(char *name)
{
	struct chiptype *p;

	for (p = chips; p->name; ++p)
		if (!strcmp(name, p->name)) {

			write_fcn    = p->w_fcn;
			amask        = p->kB * 1024 - 1;
			sector_size  = p->sector;
			write_enable = p->we;
			must_be_one  = p->mbo;
			quirks       = p->quirks;

			return (0);
		}

	return (-1);
}

static u_long
number(char *arg)
{
	char *cp;
	u_long n = strtoul(arg, &cp, 0);

	if (*cp)
		errx(2, "'%s' is not a number", arg);

	return (n);
}

/* "does not hurt" */

static void
nofault()
{
	extern char _start[], etext[], end[];
	extern char *__progname;

	char *data = (void *) &__progname;
	char *brk = sbrk(0);

	madvise(_start, etext - _start, MADV_WILLNEED);
	madvise(data,   brk - data,     MADV_WILLNEED);
}

static void
trade_powers()
{
	if (open("/dev/io", 2) == -1)
		err(1, "/dev/io");

	setuid(getuid());
}

int
main(int argc, char **argv)
{
	extern int opterr, optind;
	extern char *optarg;
	int c;

	char
		*errstr,
		*chip = NULL;
	int
		mode = 0,
		pad_end = 1;
	u_long
		n,
		x,
		size,
		start = 0,
		cnt   = 0,
		end   = 0;

	nofault();
	trade_powers();

	if (argc > 1 && argv[1][0] != '-') {
		chip = argv[1];
		optind = 2;
	}

	while ((c = getopt(argc, argv, "ibrwvs:e:n:")) != -1)
		switch (c) {
		default:
	usage:
			fprintf(stderr, "\
Usage: e2prog chip [-ibrwv][-s start][-e end -n count]\n");
			exit(2);

		/* init blankcheck read write verify */

		case 'i':
		case 'b':
		case 'r':
		case 'w':
		case 'v':
			if (mode)
				goto usage;
			mode = c;
			break;

		/* start end count */

		case 's':
			if (start)
				goto usage;
			start = number(optarg);
			break;
		case 'e':
			if (end)
				goto usage;
			end = number(optarg);
			break;
		case 'n':
			if (cnt)
				goto usage;
			cnt = number(optarg);
			break;
		}

	if (optind < argc && chip == NULL)
		chip = argv[optind++];

	if (optind != argc || !mode || (start && end && cnt))
		goto usage;

	if (lookup_chip(chip))
		errx(2, "%s unknown", chip);

	if (0 && mode == 'w' && write_fcn == w_flash) {
		product_id();
	}

	size = amask + 1;

	if (end && cnt)
		start = end - cnt;

	if (end == 0)
		end = cnt ? (start + cnt) : size;

	cnt = end - start;

	if (start >= size || end > size
	 || start >= end  || end > sizeof(file))
		errx(1, "offsets do not make sense");

	switch (mode) {

	case 'w':
	case 'v': /* write or verify, load the data file */

		if (isatty(0))
			errx(1, "will not accept input from tty");

		memset(file, 0xFF, cnt);

		n = read(0, file + start, cnt);
		if (n == -1)
			err(1, "input");
		if (n == 0)
			errx(1, "empty file");

		if (n != cnt && !pad_end)
			errx(1, "short file");

		if (read(0, file, 1) != 0)
			errx(1, "excess data in file");
	}

	enables(0);
	address(must_be_one | write_enable);

	switch (mode) {

	case 'i': /* init --- nothing to do */
		break;

	case 'b': /* blankcheck */

		for (x = start; x < end; ++x) {
			u_char
				a = rd(x);

			if (a ^ 0xFF)
				errx(1, "0x%X: chip 0x%02X", x, a);
		}
		break;

	case 'r': /* read */

		if (isatty(1))
			errx(1, "will not output to tty");

		for (x = start; x < end; ++x)
			file[x] = rd(x);

		n = write(1, file + start, cnt);
		if (n != cnt)
			err(1, "output error");

		break;

	case 'w': /* write */

		if (sector_size
		 && ((start % sector_size) || (end % sector_size)))
			errx(1, "Sector aligment");

		errstr = (*write_fcn)(start, end);

		enables(0);
		delay_ms(10);
		address(must_be_one | write_enable);

		if (errstr)
			errx(1, "%s", errstr);

		/* fall thru to ... */

	case 'v': /* verify */

		for (x = start; x < end; ++x) {
			u_char
				a = rd(x),
				b = file[x];

			if (a ^ b)
				errx(1, "0x%X: chip 0x%02X, file 0x%02X", x, a, b);
		}
		break;
	}
	exit(0);
}

static u_char
bcd(unsigned n)
{
	return (n / 10 * 16 + n % 10);
}

mk48t02(int set)
{
	if (set) {
		struct tm *tm;
		time_t t;

		time(&t);
		tm = localtime(&t);

		wr(0x7F8, 0x80); /* W */

		wr(0x7FF, bcd(tm->tm_year % 100));
		wr(0x7FE, bcd(tm->tm_mon + 1));
		wr(0x7FD, bcd(tm->tm_mday));

		wr(0x7FC, bcd(tm->tm_wday));

		wr(0x7FB, bcd(tm->tm_hour));
		wr(0x7FA, bcd(tm->tm_min));
		wr(0x7F9, bcd(tm->tm_sec));

		wr(0x7F8, 0x00); /* run */
	} else {
		printf("%02x-%02x-%02x %02x:%02x:%02x\n",
			rd(0x7FF), rd(0x7FE), rd(0x7FD),
			rd(0x7FB), rd(0x7FA), rd(0x7F9));
	}
}

at29c010a()
{
	wr(0x5555, 0xAA); /* id mode */
	wr(0x2AAA, 0x55);
	wr(0x5555, 0x90);
	delay_ms(10);

	printf("device id %02X %02X\n", rd(0x0), rd(0x1));

	printf("boot block locks %02X %02X\n", rd(0x2), rd(0x1FFF2));

	wr(0x5555, 0xAA); /* leave id mode */
	wr(0x2AAA, 0x55);
	wr(0x5555, 0xF0);
	delay_ms(10);
}

speedtest()
{
	int n = 10000;
	int i;
	struct timeval t1, t2;

	gettimeofday(&t1, 0);

	for (i = 0; i < n; ++i)
		enables(0);

	gettimeofday(&t2, 0);

	t2.tv_sec -= t1.tv_sec;
	t2.tv_usec -= t1.tv_usec;
	if (t2.tv_usec < 0) {
		t2.tv_usec += 1000000;
		t2.tv_sec--;
	}
	printf("%d enables %d.%06d second\n",
		n, t2.tv_sec, t2.tv_usec);
}

