
/* make CP/M 2.2 disk image
 * very basic...
 *
 * revision #14
 *
 *  #12 rom size tacked to wrong position if no dpb_patch
 *  #11 smacks DPB to offset 70h
 *  #10 checks exm underflow
 *
 * make_rom foo.mak
 *
 *  spt       256
 *  bytes     65536 or 32k or 8M
 *  block     1k or 2048
 *  reserve   10240 or 10k or 5t or 1M
 *  dirents   2b(locks) or 64
 *  -              ends parameters
 *  boot files
 *  -              ends boot files
 *  directory

spt 256     # 256 sectors per track
block 1k    # 1kB block
bytes 128k  # 128kB total bytes
dirents 1b  # 1 block for directory
reserve 32k # whopping 32kB for boot, monitor and such
-
bootstrap.bin
cpm22.bin
bios.bin
-
a.com
b.com

 * XXX might pack the files to the end of the image.
 * XXX might pad directory with FF, to be able to add stuff without erasure.
 * XXX pad byte. FF = rst 0x38. might not be best depending on system.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

static char *out_name = "rom.bin";

typedef unsigned char  byte;
typedef unsigned short word;

static FILE
	*out,
	*batch;

#define put(c) putc(c, out)

static long
	extent,
	dot,
	reserved,
	dpb_patch,
	bytes;

/* XXX there might be arithmetic overflows */

static word
	blocks,
	dirents,
	dirused,
	files,
	block;

static byte
	bn16bit,    /* 16 bit block numbers */
	ronly,
	hidden;

static int
	user = 0,
	spt = 256;

#define SECTOR 128 /* bytes */
#define DIRENT  32 /* bytes */

static word
	dpb_bsh, dpb_bsm, dpb_exm, dpb_dsm,
	dpb_drm, dpb_al, dpb_off;

static void
size_disk()
{
	int i;

	blocks = (bytes - reserved) / block;

	if (blocks > 0x100) {
		bn16bit = 1;
		extent  = 8 * block;
	} else {
		bn16bit = 0;
		extent  = 16 * block;
	}
	if (dirents == 0) {
		/* directory size, arbitrary XXX
		 * 1 or 4 blocks */

		if (bytes > 0x10000)
			dirents = 4 * block / DIRENT;
		else
			dirents = 1 * block / DIRENT;
	}

	/* crosscheck */

	i = 1;
	while ((SECTOR << i) < block)
		++i;

	if ((SECTOR << i) != block) {
		fprintf(stderr, "block is not %d * 2^n\n", SECTOR);
		exit(2);
	}
	if (reserved % SECTOR) {
		fprintf(stderr, "reserve is not %d * n\n", SECTOR);
		exit(2);
	}
	if (bn16bit && block < 2048) {
		fprintf(stderr, "cannot use 1k block with %lu bytes\n", bytes);
		exit(2);
	}
	dpb_bsh = i;
	dpb_bsm = (1 << i) - 1;
	dpb_exm = (block / 1024 - 1) >> bn16bit;
	dpb_dsm = blocks - 1;
	dpb_drm = dirents - 1;
	dpb_al  = ~(0xFFFFU >> (dirents / (block / DIRENT)));
	dpb_off = spt ? (int) (reserved / (spt * SECTOR)) : 0;

	printf("spt %d\nbs  %d, %d\nexm %d\ndsm %d\n"
	       "drm %d\nal  0%02Xh, 0%02Xh\ncks 0\noff %d ; %ld bytes\n",
		spt,
		dpb_bsh, dpb_bsm,
		dpb_exm,
		dpb_dsm,
		dpb_drm,
		dpb_al >> 8, dpb_al & 0xFF,
		dpb_off, reserved);

}

static void
seek_to(long offset)
{
	/* fseek forgets previous errors ? */

	if (ferror(out)) {
		perror(out_name);
		exit(1);
	}
	fseek(out, offset, SEEK_SET);
}

/* simply emit given file to output,
 * default position. return bytecount */

static long
add_data(char *name)
{
	FILE *fp;
	int c;
	long size = 0;

	fp = fopen(name, "rb");
	if (fp == NULL) {
		perror(name);
		exit(1);
	}
	while ((c = getc(fp)) != EOF) {
		put(c);
		size++;
	}
	fclose(fp);

	return (size);
}

/* append to output, start from 'dot'.
 * pad to block boundary.
 * return file size (not padded) */

static long
add_file(char *name)
{
	long pad, size;

	seek_to(dot);

	size = add_data(name);
	pad = size;
	while (pad % SECTOR) {
		put(0x1F & 'Z');
		pad++;
	}
	while (pad % block) {
		put(0xFF);
		pad++;
	}
	return (size);
}

/* add directory entry, multiple if multiple extents needed.
 * increase 'dot' to next free block.
 * return number of extents used */

static word
add_dirent(char *name, long size)
{
	int i;
	word ex = 0;
	byte namext[8+3];
	char *ext = strchr(name, '.');

	if (ext)
		ext++;
	else
		ext = "";

	for (i = 0; i < 8; ++i)
		namext[i] = (*name && *name != '.') ? toupper(*name++) : 0x20;

	for (i = 0; i < 3; ++i)
		namext[8+i] = *ext ? toupper(*ext++) : 0x20;

	namext[8] |= ronly;
	namext[9] |= hidden;

	seek_to(reserved + DIRENT * dirused);

	do {
		put(user);
		for (i = 0; i < 8+3; ++i)
			put(namext[i]);

		/* EX S1 S2 RC
		 * XXX 32? though, not many extents likely to be used */

		put(ex % 32);
		put(0x00);
		put(ex / 32);

		if (size >= extent)
			put(extent / SECTOR);
		else
			put((size + SECTOR - 1) / SECTOR);

		/* allocation, 16 bytes or 8 words */

		i = 0;
		do {
			word bn = 0;

			if (size) {
				if (size >= block)
					size -= block;
				else
					size = 0;

				/* relative to reserved */

				bn = (dot - reserved) / block;
				dot += block;
			}
			put(bn);
			i++;
			if (bn16bit) {
				put(bn >> 8);
				i++;
			}
		} while (i < 16);
		ex++;
	} while (size);

	return (ex);
}

/* ordering is important,
 * if unit refers another (previous) parameter */

static void
command(char *cmd)
{
	char *rest, *unit;

	/* split command and rest */

	for (rest = cmd; *rest; ++rest) {
		if (isspace(*rest)) {
			*rest++ = 0;
			while (*rest && isspace(*rest))
				rest++;
			break;
		}
	}

	if (!strcmp(cmd, "block")) {
		block = strtoul(rest, &unit, 10);
		if (*unit == 'k')
			block *= 1024;
		return;
	}
	if (!strcmp(cmd, "bytes")) {
		bytes = strtoul(rest, &unit, 10);
		if (*unit == 'k')
			bytes *= 1024;
		if (*unit == 'M')
			bytes *= 1024 * 1024;
		if (*unit == 't')
			bytes *= spt * SECTOR; /* tracks */
		return;
	}
	if (!strcmp(cmd, "dirents")) {
		dirents = strtoul(rest, &unit, 10);
		if (*unit == 'b')
			dirents *= block / DIRENT; /* blocks */
		return;
	}
	if (!strcmp(cmd, "spt")) {
		spt = strtoul(rest, &unit, 10); /* just convenience for "reserved" */
		return;
	}
	if (!strcmp(cmd, "reserve")
	 || !strcmp(cmd, "reserved")) {
		reserved = strtoul(rest, &unit, 10);
		if (*unit == 't')
			reserved *= spt * SECTOR; /* tracks */
		if (*unit == 'k')
			reserved *= 1024;
		if (*unit == 'M')
			reserved *= 1024 * 1024;
		return;
	}

	fprintf(stderr, "command '%s' unknown\n", cmd);
	exit(2);
}

/* return next non-comment nonempty line or NULL */

static int
getline(char **p)
{
	static char buf[256];
	static int lnum;
	char *cp;

	while (fgets(buf, sizeof(buf), batch)) {
		lnum++;
		cp = strchr(buf, '\n');
		if (cp == NULL) {
			fprintf(stderr, "line %d too long\n", lnum);
			exit(2);
		}
		while (cp > buf && isspace(cp[-1]))
			--cp;
		*cp = 0;

		cp = buf;
		while (*cp && isspace(*cp))
			cp++;

		if (*cp == '#' || *cp == 0)
			continue;

		*p = cp;
		return (1);
	}
	return (0);
}

int
main(int argc, char **argv)
{
	char *line, *file;
	long reserve;

	if (argc != 2) {
		fprintf(stderr, "Usage: make_rom rom.mak\n");
		exit(2);
	}
	batch = fopen(argv[1], "r");
	if (batch == NULL) {
		perror(argv[1]);
		exit(1);
	}

	/* set parameters */

	while (getline(&line)) {
		if (*line == '-')
			break;

		command(line);
	}

	size_disk();

	out = fopen(out_name, "wb");
	if (out == NULL) {
		perror(out_name);
		exit(1);
	}

	/* reserved area. just pack from files, until - seen */

	while (getline(&file)) {
		if (*file == '-')
			break;

		if (!strcmp(file, "dpb")) {
			dpb_patch = dot;
			continue;
		}
		if (!memcmp(file, "org", 3)) {
			char *p;
			long arg = strtoul(file + 3, &p, 0);

			if (*p == 0) {
				if (dot > arg) {
					fprintf(stderr, "org 0x%lX is backwards\n", arg);
					exit(1);
				}
				while (dot < arg) {
					put(0xFF);
					dot++;
				}
				continue;
			}
		}

		dot += add_data(file);
	}
	reserve = reserved - dot;

	if (dot > reserved) {
		fprintf(stderr, "reserve overflow, %ld\n", dot);
		exit(1);
	}

	/* fill over reserved area, FF */

	while (dot < reserved) {
		put(0xFF);
		dot++;
	}

	/* fill over directory, E5 */

	while (dot < reserved + dirents * DIRENT) {
		put(0xE5);
		dot++;
	}

	/* filesystem. rest of files */

	while (getline(&file)) {
		long size;
		char *basename, *slash;

		if (!memcmp(file, ".user", 5)) {
			char *p;
			long arg = strtoul(file + 5, &p, 0);

			if (*p == 0) {
				user = arg;
				continue;
			}
		}
		if (!strcmp(file, ".hidden")) {
			hidden ^= 0x80;
			continue;
		}
		if (!memcmp(file, ".ronly", 4)) {
			ronly ^= 0x80;
			continue;
		}
		/* fishing XXX drive letter and other nasties */

		basename = file;
		slash = strrchr(basename,
#ifdef unix
				'/'
#else
				'\\'
#endif
			);
		if (slash)
			basename = slash + 1;

		size = add_file(file);
		dirused += add_dirent(basename, size);
		files++;
	}
	fclose(batch);

	if (dot > bytes) {
		fprintf(stderr, "data overflow, 0x%04lX\n", dot);
		exit(1);
	}
	if (dirused > dirents) {
		fprintf(stderr, "directory overflow, %d\n", dirused);
		exit(1);
	}
	if (dpb_patch) {
		seek_to(dpb_patch); /* XXX */

		put(spt);
		put(spt >> 8);
		put(dpb_bsh);
		put(dpb_bsm);
		put(dpb_exm);
		put(dpb_dsm);
		put(dpb_dsm >> 8);
		put(dpb_drm);
		put(dpb_drm >> 8);
		put(dpb_al >> 8);    /* AL0 AL1 is not a word */
		put(dpb_al);
		put(0);
		put(0);
		put(dpb_off);
		put(dpb_off >> 8);

		seek_to(dpb_patch + 127); /* XXX */

		put(bytes / (32 * 1024));
	}

	if (ferror(out) || fclose(out)) {
		perror(out_name);
		exit(1);
	}
	printf("%s, %d files, space left %ld. reserve left %ld\n",
		out_name, files, bytes - dot, reserve);

	exit(0);
}

